import sqlite3
import json
import time
from datetime import datetime, timedelta
import threading
import uuid
# ------------------------------------------------------------
# ⚙️ الإعدادات والثوابت
# ------------------------------------------------------------
DB_FILE = r"C:\Users\Administrator\Desktop\ichancybots\testbot\testluf\users.db"
DB_LOCK = threading.RLock()

# ------------------------------------------------------------
# 🗄️ مدير الاتصال (Context Manager)
# ------------------------------------------------------------

class SQLiteConnection:
    """يضمن فتح الاتصال والتزام التغييرات أو التراجع عنها، وإغلاق الاتصال."""
    def __init__(self, db_file):
        self.db_file = db_file
        self.conn = None

    def __enter__(self):
        self.conn = sqlite3.connect(self.db_file)
        self.conn.row_factory = sqlite3.Row
        # تمكين WAL mode لتحسين التزامن
        self.conn.execute("PRAGMA journal_mode=WAL;") 
        return self.conn

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.conn.commit()
        else:
            self.conn.rollback()
        self.conn.close()
        
def get_connection():
    return SQLiteConnection(DB_FILE)

# ------------------------------------------------------------
# 🏗️ تهيئة قاعدة البيانات (إنشاء الجداول)
# ------------------------------------------------------------

# كود مؤقت لتصحيح البيانات القديمة


def init_db():
    """إنشاء الجداول المطلوبة وتعبئة الإعدادات الافتراضية."""
    
    # ⚠️ نستخدم الاتصال التقليدي هنا لأننا خارج السياق الطبيعي للدوال
    conn = sqlite3.connect(DB_FILE)
    conn.row_factory = sqlite3.Row
    conn.execute("PRAGMA journal_mode=WAL;")
    cur = conn.cursor()

    # 1. جدول المستخدمين (user_accounts)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS user_accounts (
            user_id TEXT PRIMARY KEY,
            username TEXT,
            password TEXT,
            balance REAL DEFAULT 0.0,
            total_deposited REAL DEFAULT 0.0,
            bonus_balance REAL DEFAULT 0.0,
            usdt_balance REAL DEFAULT 0.0,
            has_withdrawn INTEGER DEFAULT 0, -- 0 for False, 1 for True
            referral_pending REAL DEFAULT 0.0,
            referral_available REAL DEFAULT 0.0,
            referral_total REAL DEFAULT 0.0,
            last_wheel_spin INTEGER DEFAULT 0,
            daily_wheel_day INTEGER DEFAULT 0,
            daily_wheel_spins_used INTEGER DEFAULT 0,
            user_history TEXT DEFAULT '[]', -- حفظ التاريخ كـ JSON داخل الحقل
            pending_referrals TEXT DEFAULT NULL,
            wheel_last_day INTEGER DEFAULT 0,
            wheel_spins_used INTEGER DEFAULT 0
        )
    """)
    
    try:
        cur.execute("ALTER TABLE user_accounts ADD COLUMN password TEXT")
    except sqlite3.OperationalError:
        pass  # العمود موجود مسبقًا
    cur.execute("""
        CREATE TABLE IF NOT EXISTS promo_codes (
            code TEXT PRIMARY KEY,
            value REAL,
            max_uses INTEGER,
            uses_left INTEGER,
            published INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            extra TEXT DEFAULT '{}'
        )
    """)
    try:
        cur.execute("ALTER TABLE promo_codes ADD COLUMN extra TEXT DEFAULT '{}'")
    except sqlite3.OperationalError:
        pass
    try:
        cur.execute("ALTER TABLE promo_codes ADD COLUMN published INTEGER DEFAULT 0")
    except sqlite3.OperationalError:
        pass
    cur.execute("""
        CREATE TABLE IF NOT EXISTS promo_code_usage (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            code TEXT,
            user_id TEXT,
            used_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(code, user_id)
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS pending_deposits (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id TEXT NOT NULL,
            amount REAL NOT NULL,
            operation_code TEXT UNIQUE NOT NULL,
            method TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)

    
    # 2. جدول سجلات العمليات (transaction_logs)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS transaction_logs (
            id INTEGER PRIMARY KEY,
            transaction_id TEXT UNIQUE,
            user_id TEXT,
            type TEXT, 
            amount REAL,
            method TEXT,
            status TEXT,
            extra TEXT,
            client_number TEXT,
            requested_amount REAL,
            sent_amount REAL,
            fee_percent REAL,  
            timestamp REAL,
            FOREIGN KEY(user_id) REFERENCES user_accounts(user_id)
        )
    """)
    try:
        cur.execute("ALTER TABLE transaction_logs ADD COLUMN transaction_id TEXT UNIQUE")
    except sqlite3.OperationalError:
        pass
        
    try:
        cur.execute("ALTER TABLE transaction_logs ADD COLUMN requested_amount REAL DEFAULT 0.0")
    except sqlite3.OperationalError:
        pass
        
    try:
        cur.execute("ALTER TABLE transaction_logs ADD COLUMN sent_amount REAL DEFAULT 0.0")
    except sqlite3.OperationalError:
        pass
        
    try:
        cur.execute("ALTER TABLE transaction_logs ADD COLUMN fee_percent REAL DEFAULT 0.0")
    except sqlite3.OperationalError:
        pass
        
    try:
        cur.execute("ALTER TABLE transaction_logs ADD COLUMN status TEXT DEFAULT 'pending'")
    except sqlite3.OperationalError:
        pass
    
    # 3. جدول الإعدادات العامة (app_settings)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS app_settings (
            key TEXT PRIMARY KEY,
            value TEXT
        )
    """)
    
    # 4. جدول الإحالات والعلاقات
    cur.execute("""
        CREATE TABLE IF NOT EXISTS referral_relationships (
            invited_id TEXT PRIMARY KEY,
            referrer_id TEXT,
            referral_timestamp INTEGER DEFAULT 0
            
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS system_settings (
            key TEXT PRIMARY KEY,
            value TEXT
        )
    """)
        # 5. جدول طلبات السحب المعلقة (pending_withdrawals)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS pending_withdrawals (
            request_id TEXT PRIMARY KEY,
            user_id TEXT NOT NULL,
            client_number TEXT NOT NULL,
            requested_amount REAL NOT NULL,
            original_currency TEXT DEFAULT 'SYP',
            method TEXT NOT NULL,
            timestamp REAL NOT NULL,
            amount_in_usdt REAL DEFAULT 0.0,
            network TEXT DEFAULT '',
            wallet_address TEXT DEFAULT '',
            status TEXT DEFAULT 'pending'
        )
    """)
    
    
    # 5. جدول أرباح الإحالة
    cur.execute("""
        CREATE TABLE IF NOT EXISTS referral_earnings (
            id TEXT PRIMARY KEY, -- (مثل: hourly_invitedId_referrerId_level)
            referrer_id TEXT,
            invited_id TEXT,
            amount REAL,
            status TEXT, -- pending, available
            level INTEGER,
            created_at TEXT
        )
    """)
    
    # 6. جدول سجلات المدراء (Admin History)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS admin_history (
            id INTEGER PRIMARY KEY,
            user_identifier TEXT,
            type TEXT, 
            amount REAL,
            method TEXT,
            status TEXT, 
            extra TEXT, -- لحفظ البيانات الإضافية كـ JSON
            timestamp TEXT
        )
    """)
    # 8. جدول المستخدمين المحظورين
    cur.execute("""
        CREATE TABLE IF NOT EXISTS banned_users (
            user_id TEXT PRIMARY KEY
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS user_referral_summary (
            user_id TEXT PRIMARY KEY,
            referral_pending REAL DEFAULT 0.0,
            referral_available REAL DEFAULT 0.0,
            referral_total REAL DEFAULT 0.0,
            last_processed_deposit_id TEXT DEFAULT NULL
        )
    """)

    # 7. تعبئة الإعدادات الافتراضية
    settings_defaults = {
        "cashier_profit": "0.0",
        "exchange_rate": "4500",
        "support_username": "Matador_Support",
        "bot_status": "running",
        "deposit_withdraw_enabled": "True",
        "last_referral_distribution": None,
        # يمكن إضافة باقي الإعدادات العامة هنا
    }
    for key, default_value in settings_defaults.items():
        value_str = str(default_value) if default_value is not None else 'null'
        cur.execute("INSERT OR IGNORE INTO app_settings (key, value) VALUES (?, ?)", (key, value_str))
    
    # بعد conn.commit() وقبل conn.close()

    # ⬇️ إضافة الأعمدة المفقودة إذا لم تكن موجودة (SQLite لا يدعم ALTER ADD COLUMN IF NOT EXISTS)
    try:
        cur.execute("ALTER TABLE user_accounts ADD COLUMN wheel_last_day INTEGER DEFAULT 0")
    except sqlite3.OperationalError:
        pass  # العمود موجود مسبقًا

    try:
        cur.execute("ALTER TABLE user_accounts ADD COLUMN wheel_spins_used INTEGER DEFAULT 0")
    except sqlite3.OperationalError:
        pass  # العمود موجود مسبقًا

    conn.commit()
    conn.close()

init_db() 

# ------------------------------------------------------------
# 🎯 دوال التعامل مع الإعدادات العامة
# ------------------------------------------------------------

def _get_setting(key: str, default=None):
    """جلب قيمة إعداد من app_settings."""
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("SELECT value FROM app_settings WHERE key = ?", (key,))
        row = cur.fetchone()
        if row and row["value"] != 'null':
            # محاولة تحويل القيمة إذا كانت رقمية أو منطقية
            val = row["value"]
            try:
                if val.lower() == 'true': return True
                if val.lower() == 'false': return False
                return float(val) if '.' in val or 'e' in val else int(val)
            except ValueError:
                return val
        return default

def _set_setting(key: str, value):
    """تحديث قيمة إعداد في app_settings."""
    value_str = str(value) if value is not None else 'null'
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("INSERT OR REPLACE INTO app_settings (key, value) VALUES (?, ?)", (key, value_str))

def get_bot_status() -> str:
    return _get_setting("bot_status", "running")

def set_bot_status(status: str):
    _set_setting("bot_status", status)

def get_support_username():
    """
    جلب اسم مستخدم الدعم من قاعدة البيانات.
    يبحث أولاً في system_settings، ثم في app_settings.
    """
    print("🔍 get_support_username() called")
    
    # 1. جرب من system_settings أولاً
    from database import get_system_setting
    username = get_system_setting('support_username')
    print(f"   من system_settings: '{username}'")
    
    # 2. إذا لم يكن في system_settings، جرب من app_settings
    if not username or username in ['null', '', 'None']:
        username = _get_setting('support_username', 'matadorsupport')
        print(f"   من app_settings: '{username}'")
    
    # 3. تنظيف النتيجة
    if username in [None, 'null', '', 'None']:
        username = 'matadorsupport'
    
    print(f"✅ النتيجة النهائية: '{username}'")
    return str(username)

def get_support_link():
    username = get_support_username()
    return f"https://t.me/{username}"

def toggle_deposit_withdraw():
    current = _get_setting("deposit_withdraw_enabled", True)
    _set_setting("deposit_withdraw_enabled", not current)
    return not current

# ------------------------------------------------------------
# 👤 دوال التعامل مع المستخدمين
# ------------------------------------------------------------

def create_local_user(user_id: str, username: str = None, password: str = None) -> bool:
    """إنشاء مستخدم جديد في جدول user_accounts."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # التحقق من وجود المستخدم
            cur.execute("SELECT user_id FROM user_accounts WHERE user_id = ?", (user_id,))
            if cur.fetchone():
                print(f"ℹ️ المستخدم {user_id} موجود مسبقاً")
                return False  # المستخدم موجود بالفعل
            
            # إضافة المستخدم
            default_username = username or f"user{user_id}"
            try:
                # ✅ تصحيح: إضافة حقل password إلى الاستعلام
                cur.execute("""
                    INSERT INTO user_accounts (user_id, username, password, user_history) 
                    VALUES (?, ?, ?, ?)
                """, (user_id, default_username, password, '[]'))
                conn.commit()
                print(f"✅ تم إنشاء حساب للمستخدم {user_id} مع الباسورد: {password}")
                return True
            except sqlite3.Error as e:
                print(f"❌ فشل إنشاء حساب للمستخدم {user_id}: {e}")
                return False
                
def get_user_data(user_id: str) -> dict:
    """جلب بيانات المستخدم كاملة."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT * FROM user_accounts WHERE user_id = ?", (user_id,))
            row = cur.fetchone()
            if row:
                data = dict(row)
                data['has_withdrawn'] = bool(data['has_withdrawn'])
                data['user_history'] = json.loads(data['user_history'])
                return data
            return {}

def update_user_field(user_id: str, field: str, value):
    """تحديث حقل واحد للمستخدم."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            if field == 'has_withdrawn':
                value = 1 if value else 0
            
            cur.execute(f"UPDATE user_accounts SET {field} = ? WHERE user_id = ?", (value, user_id))


def add_user_history(user_id, operation_type, amount, method):
    """إضافة سجل لـ user_history كـ JSON."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # جلب التاريخ الحالي
            cur.execute("SELECT user_history FROM user_accounts WHERE user_id = ?", (user_id,))
            row = cur.fetchone()
            if not row: return

            history = json.loads(row["user_history"])
            history.append({
                "type": operation_type,
                "amount": float(amount),
                "method": method,
                "timestamp": str(datetime.now())
            })
            
            # تحديث حقل user_history
            cur.execute("UPDATE user_accounts SET user_history = ? WHERE user_id = ?", (json.dumps(history), user_id))

# ------------------------------------------------------------
# 💸 دوال العمليات (Transactions)
# ------------------------------------------------------------


def add_transaction(user_id, tx_type, amount, method, status="success", client_number=None, metadata=None):
    """إضافة سجل معاملة وتحديث رصيد المستخدم مع منع الرصيد السالب."""
    amount = float(amount)
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 1. التحقق من الرصيد في حالة السحب
            if tx_type == "withdraw":
                cur.execute("SELECT balance FROM user_accounts WHERE user_id = ?", (user_id,))
                row = cur.fetchone()
                if row and float(row['balance']) < amount:
                    print(f"❌ محاولة سحب {amount} برصيد غير كافٍ: {row['balance']}")
                    return False
            
            # 2. إضافة سجل المعاملة مع metadata في حقل extra
            cur.execute("""
                INSERT INTO transaction_logs (user_id, type, amount, method, status, timestamp, client_number, extra) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            """, (
                user_id, 
                tx_type, 
                amount, 
                method, 
                status, 
                time.time(), 
                client_number,
                json.dumps(metadata) if metadata else None  # ✅ تحويل metadata إلى JSON
            ))

            # 3. تحديث الرصيد مع التحقق من عدم السالب
            if tx_type in ["deposit", "wheel_win", "bonus", "gift_receive", "website_withdraw"]:
                # زيادة الرصيد - لا مشكلة هنا
                update_sql = "UPDATE user_accounts SET balance = balance + ? WHERE user_id = ?"
                cur.execute(update_sql, (amount, user_id))
                
            elif tx_type in ["withdraw", "gift_send"]:
                # في حالة السحب: الخصم مع التحقق
                cur.execute("""
                    UPDATE user_accounts 
                    SET balance = balance - ? 
                    WHERE user_id = ? AND balance >= ?
                """, (abs(amount), user_id, abs(amount)))
                
                # التحقق من نجاح العملية
                if cur.rowcount == 0:
                    print(f"❌ فشل السحب: رصيد غير كافٍ أو خطأ للمستخدم {user_id}")
                    conn.rollback()
                    return False
            elif tx_type == "website_deposit":
                # خصم من الرصيد عند الإيداع على الموقع
                cur.execute("""
                    UPDATE user_accounts 
                    SET balance = balance - ? 
                    WHERE user_id = ? AND balance >= ?
                """, (amount, user_id, amount))
                
                if cur.rowcount == 0:
                    print(f"❌ فشل website_deposit: رصيد غير كافٍ للمستخدم {user_id}")
                    conn.rollback()
                    return False
            
            # 4. تحديث إجمالي الإيداعات
            if tx_type == "deposit" and status == "success":
                update_sql = "UPDATE user_accounts SET total_deposited = total_deposited + ? WHERE user_id = ?"
                cur.execute(update_sql, (amount, user_id))
            
            conn.commit()
            return True

def get_net_deposit_last_24h(user_id: str):
    now_ts = time.time()
    cutoff_ts = now_ts - (24 * 3600)

    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT 
                    COALESCE(SUM(
                        CASE 
                            WHEN type = 'deposit' AND status = 'success' THEN amount 
                            ELSE 0 
                        END
                    ), 0) AS deposits,
                    COALESCE(SUM(
                        CASE 
                            WHEN type = 'withdraw' AND status = 'success' THEN amount 
                            ELSE 0 
                        END
                    ), 0) AS withdrawals
                FROM transaction_logs
                WHERE user_id = ?
                  AND timestamp >= ?
            """, (user_id, cutoff_ts))

            row = cur.fetchone()
            if not row:
                return 0.0

            return float(row['deposits']) - float(row['withdrawals'])

# ------------------------------------------------------------
# 👑 دوال سجلات المدراء (Admin History)
# ------------------------------------------------------------

def add_admin_history(user_identifier, operation_type, amount, method, status="success", extra=None):
    """إضافة سجل لعمليات المدراء."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            extra_json = json.dumps(extra) if extra else None
            
            cur.execute("""
                INSERT INTO admin_history (user_identifier, type, amount, method, status, extra, timestamp) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """, (
                user_identifier, 
                operation_type, 
                float(amount) if amount is not None else None, 
                method, 
                status, 
                extra_json,
                str(datetime.now())
            ))

# ------------------------------------------------------------
# 🛑 المهاجرة والتنظيف (Migration and Cleanup)
# ------------------------------------------------------------
# يتم تجاهل دوال التحميل القديمة مثل load_data, save_data, load_db, save_all_changes
# لأنها لم تعد ضرورية مع النظام الجديد.

def migrate_transaction_logs():
    """تحويل السجلات القديمة لتحديد النوع الصحيح."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 1. تحديث العمليات التي كانت withdraw لكن نوعها غير محدد
            cur.execute("""
                UPDATE transaction_logs
                SET type = 'withdraw'
                WHERE type IS NOT 'withdraw' AND method = 'manual_withdraw' AND status = 'success'
            """)
            migrated_count = cur.rowcount

            # 2. تحديث العمليات التي ليس لها نوع لـ 'deposit' كافتراضي
            cur.execute("""
                UPDATE transaction_logs
                SET type = 'deposit'
                WHERE type IS NULL OR type = ''
            """)
            migrated_count += cur.rowcount
            
            # 3. التأكد من حالة و timestamp
            cur.execute("UPDATE transaction_logs SET status = 'success' WHERE status IS NULL")
            cur.execute("UPDATE transaction_logs SET timestamp = ? WHERE timestamp IS NULL", (time.time(),))

            return migrated_count
            

# تنفيذ المهاجرة بعد التهيئة
migrate_transaction_logs()

# ⚠️ تم حذف الدوال القديمة:
# - initialize_pending_referrals
# - add_initial_deposit 
# واستبدالها بالدوال الجديدة التي تستخدم SQLite مباشرة. 
# يجب دمج منطق initialize_pending_referrals في الدالة create_local_user إذا لزم الأمر.

def init_used_transactions_table():
    """إنشاء جدول لتخزين أرقام العمليات التي تم استخدامها لمنع التكرار."""
    conn = sqlite3.connect(DB_FILE)
    conn.execute("PRAGMA journal_mode=WAL;")
    cur = conn.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS used_transactions (
            reference_id TEXT PRIMARY KEY, -- رقم العملية أو TX ID
            user_id TEXT,
            amount REAL,
            method TEXT,
            timestamp REAL
        )
    """)
    conn.commit()
    conn.close()

# تنفيذ التهيئة
init_used_transactions_table()


async def check_used_transaction(reference_id: str) -> bool:
    """التحقق مما إذا كان رقم العملية (reference_id) قد تم استخدامه مسبقاً."""
    # (تم استخدام async للمحافظة على توقيع الدالة في receive_deposit_proof)
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT 1 FROM used_transactions WHERE reference_id = ?", (reference_id,))
            return cur.fetchone() is not None

async def add_used_transaction(reference_id: str, amount: float, method: str, user_id: str):
    """إضافة رقم عملية إلى سجل العمليات المستخدمة."""
    # (تم استخدام async للمحافظة على توقيع الدالة في receive_deposit_proof)
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                INSERT OR IGNORE INTO used_transactions (reference_id, user_id, amount, method, timestamp) 
                VALUES (?, ?, ?, ?, ?)
            """, (reference_id, user_id, amount, method, time.time()))
            
# ------------------------------------------------------------
# 💰 دوال التحديث المباشر للرصيد
# ------------------------------------------------------------

async def update_user_balance(user_id: str, total_amount: float, deposited_amount: float):
    """تحديث رصيد المستخدم وإجمالي الإيداعات."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            # زيادة الرصيد (Total Amount = Base + Bonus)
            cur.execute(
                "UPDATE user_accounts SET balance = balance + ? WHERE user_id = ?",
                (total_amount, user_id)
            )
            # تحديث إجمالي الإيداعات (Base Amount)
            cur.execute(
                "UPDATE user_accounts SET total_deposited = total_deposited + ? WHERE user_id = ?",
                (deposited_amount, user_id)
            )

async def add_transaction_log(user_id: str, tx_type: str, method: str, amount: float, bonus: float, client_number: str = None):
    """إضافة سجل مفصل للعملية في transaction_logs."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            # status='success' افتراضياً لعمليات الشحن المكتملة
            cur.execute("""
                INSERT INTO transaction_logs (user_id, type, amount, method, status, timestamp, client_number, extra) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            """, (
                user_id, tx_type, amount, method, 'success', time.time(), client_number, 
                json.dumps({'bonus': bonus}) # حفظ البونص في حقل إضافي
            ))
            
# ⚠️ ملاحظة: يجب تعديل جدول transaction_logs لإضافة حقل 'extra' أو استخدام حقل 'client_number' لحفظ البونص إذا لزم الأمر.
# إذا لم ترغب بتعديل الجدول، يمكنك إزالة الإشارة إلى 'extra' في الدالة وتجاهل تسجيل البونص ضمن transaction_logs.
def delete_user(user_id_str: str) -> bool:
    """حذف جميع سجلات المستخدم من جدول user_accounts."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("DELETE FROM user_accounts WHERE user_id = ?", (user_id_str,))
            return cur.rowcount > 0
        
        
# في ملف database.py

def get_user_daily_wheel_usage(user_id_str: str) -> tuple[int, int]:
    """
    جلب عدد الدورات المستخدمة (spins_used) وآخر يوم تقويمي (last_day) من DB.

    العودة: (last_day, spins_used)
    """
    with get_connection() as conn:
        cur = conn.cursor()
        # ⚠️ ملاحظة: نفترض وجود جدول 'user_wheel_stats' أو حقول في 'user_accounts'
        # هنا سنفترض وجود حقلين في جدول 'user_accounts'
        cur.execute("SELECT wheel_last_day, wheel_spins_used FROM user_accounts WHERE user_id = ?", (user_id_str,))
        row = cur.fetchone()
        if row and row['wheel_last_day'] is not None:
            # wheel_last_day, wheel_spins_used
            return (row['wheel_last_day'], row['wheel_spins_used'])
        else:
            return (0, 0)
        
def update_user_daily_wheel_usage(user_id_str: str, new_day: int, new_spins: int):
    now_ts = int(time.time())
    """
    تحديث عدد دورانات العجلة المستخدمة وآخر يوم تقويمي في جدول user_accounts.

    :param user_id_str: معرف المستخدم (معرّف Telegram).
    :param new_day: رقم اليوم التقويمي الحالي (الجديد).
    :param new_spins: إجمالي عدد الدورات المستخدمة لهذا اليوم.
    """
    try:
        # 🔒 استخدام القفل لتجنب تعارض الكتابة
        with DB_LOCK:
            with get_connection() as conn:
                cur = conn.cursor()
                
                # 📝 استعلام التحديث: يحدد المستخدم ويُحدِّث حقلي العجلة
                cur.execute(
                    """
                    UPDATE user_accounts 
                    SET wheel_last_day = ?, 
                        wheel_spins_used = ? ,
                        last_wheel_spin = ?
                    WHERE user_id = ?
                    """,
                    (new_day, new_spins, now_ts, user_id_str))
                
                conn.commit()
                
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to update wheel usage for user {user_id_str}: {e}")
        # يمكنك إضافة معالجة خطأ أخرى أو رفع استثناء هنا
        
def get_referral_relationships() -> dict[str, str]:
    """
    جلب جميع علاقات الإحالة من جدول referral_relationships.
    العودة: قاموس بالشكل {invited_id: referrer_id}
    """
    relationships = {}
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT invited_id, referrer_id FROM referral_relationships")
            rows = cur.fetchall()
            for row in rows:
                invited_id = str(row['invited_id'])
                referrer_id = str(row['referrer_id'])
                relationships[invited_id] = referrer_id
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch referral relationships: {e}")
        return {}
    return relationships
def get_user_transactions(user_id_str: str, tx_type: str = None) -> list[dict]:
    """
    جلب سجلات المعاملات لمستخدم معين ونوع معاملة محدد.
    """
    logs = []
    
    # بناء الاستعلام - ⚠️ تم التصحيح: FROM transaction_logs بدلاً من transactions
    query = "SELECT amount, method, status, timestamp, type FROM transaction_logs WHERE user_id = ?"
    params = (user_id_str,)
    
    if tx_type:
        query += " AND type = ?"
        params += (tx_type,)
        
    query += " ORDER BY timestamp DESC"

    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(query, params)
            
            rows = cur.fetchall()
            
            for row in rows:
                logs.append({
                    "amount": row['amount'],
                    "method": row['method'],
                    "status": row['status'],
                    "timestamp": row['timestamp'],
                    "type": row['type'],
                })
                
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch transactions for user {user_id_str}: {e}")
        
    return logs
# في database.py
def get_first_layer_referrals_count(referrer_id):
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT COUNT(*) 
                FROM referral_relationships 
                WHERE referrer_id = ?
            """, (str(referrer_id),))
            return cur.fetchone()[0] or 0
    except Exception as e:
        print(f"❌ خطأ في حساب عدد الإحالات المباشرة: {e}")
        return 0

def get_user_net_balance_from_transactions(user_id_str: str) -> float:
    """
    حساب صافي الرصيد (مجموع الإيداعات الناجحة - مجموع السحوبات الناجحة)
    لغرض حساب عمولة الإحالة.
    """
    net_deposit = 0.0
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 1. مجموع الإيداعات الناجحة - ✅ تصحيح: إضافة FROM
            cur.execute(
                """
                SELECT SUM(amount) FROM transaction_logs
                WHERE user_id = ? AND type = 'deposit' AND status = 'success'
                """,
                (user_id_str,)
            )
            total_deposit_row = cur.fetchone()
            total_deposit = total_deposit_row[0] if total_deposit_row and total_deposit_row[0] else 0.0
            
            # 2. مجموع السحوبات الناجحة - هذا صحيح
            cur.execute(
                """
                SELECT SUM(amount) FROM transaction_logs
                WHERE user_id = ? AND type = 'withdraw' AND status = 'success'
                """,
                (user_id_str,)
            )
            total_withdraw_row = cur.fetchone()
            total_withdraw = total_withdraw_row[0] if total_withdraw_row and total_withdraw_row[0] else 0.0
            
            net_deposit = max(total_deposit - total_withdraw, 0.0)
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to calculate net balance for {user_id_str}: {e}")
        
    return net_deposit
# ⚠️ يجب توفير الدوال الأخرى المذكورة أعلاه (get_user_data, get_referral_relationships, إلخ)
# وتعديلها للعمل مع SQLite في ملف database.py.

def get_all_referral_earnings_by_referrer(referrer_id_str: str) -> list[dict]:
    """
    جلب جميع سجلات أرباح الإحالة التي قام بها هذا المستخدم (كمُحيل).
    """
    earnings = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                SELECT referrer_id, invited_id, amount, status, level, created_at 
                FROM referral_earnings 
                WHERE referrer_id = ? 
                ORDER BY created_at DESC
                """,
                (referrer_id_str,)
            )
            rows = cur.fetchall()
            
            for row in rows:
                earnings.append(dict(row))
                
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch earnings for referrer {referrer_id_str}: {e}")
        
    return earnings

def get_last_referral_distribution() -> str | None:
    """
    جلب تاريخ آخر توزيع لأرباح الإحالة من جدول system_settings.
    العودة: تاريخ بتنسيق ISO 8601 كنص (str) أو None.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT value FROM system_settings WHERE key = 'last_referral_distribution'"
            )
            row = cur.fetchone()
            return row['value'] if row else None
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get last referral distribution date: {e}")
        return None

def set_last_referral_distribution(date_str: str):
    """
    تعيين تاريخ آخر توزيع لأرباح الإحالة في جدول system_settings.
    :param date_str: التاريخ بتنسيق ISO 8601.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            # استخدام INSERT OR REPLACE لضمان التحديث إذا كان المفتاح موجودًا
            cur.execute(
                """
                INSERT OR REPLACE INTO system_settings (key, value) 
                VALUES ('last_referral_distribution', ?)
                """,
                (date_str,)
            )
            conn.commit()
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to set last referral distribution date: {e}")
        
        
# ⚠️ يجب أن يكون DAYS_TO_UNLOCK معرفاً، لنفترض أنه 30 يوماً
DAYS_TO_UNLOCK = 30 


def get_user_referral_earnings_summary(referrer_id_str: str) -> dict:
    summary = {'total': 0.0, 'pending': 0.0, 'available': 0.0, 'next_unlock': None}
    now = datetime.now()

    try:
        with get_connection() as conn:
            cur = conn.cursor()

            # المبالغ المعلقة بالكامل
            cur.execute("""
                SELECT SUM(amount) as pending_total FROM referral_earnings 
                WHERE referrer_id = ? AND status = 'pending'
            """, (referrer_id_str,))
            summary['pending'] = cur.fetchone()['pending_total'] or 0.0

            # المبالغ الجاهزة للسحب (10+ يوم)
            cutoff_date = (now - timedelta(days=DAYS_TO_UNLOCK)).isoformat()
            cur.execute("""
                SELECT SUM(amount) as available_total FROM referral_earnings 
                WHERE referrer_id = ? AND status = 'pending' AND created_at <= ?
            """, (referrer_id_str, cutoff_date))
            summary['available'] = cur.fetchone()['available_total'] or 0.0

            # المبالغ الموزعة والمتاحة سابقًا
            cur.execute("""
                SELECT SUM(amount) as distributed_total FROM referral_earnings 
                WHERE referrer_id = ? AND status IN ('available', 'distributed')
            """, (referrer_id_str,))
            distributed_total = cur.fetchone()['distributed_total'] or 0.0

            summary['total'] = summary['pending'] + distributed_total

            # أقرب تاريخ تحرير
            if summary['pending'] > 0:
                cur.execute("""
                    SELECT created_at FROM referral_earnings 
                    WHERE referrer_id = ? AND status = 'pending'
                    ORDER BY created_at ASC LIMIT 1
                """, (referrer_id_str,))
                row = cur.fetchone()
                if row:
                    created_at = datetime.fromisoformat(row['created_at'])
                    summary['next_unlock'] = (created_at + timedelta(days=DAYS_TO_UNLOCK)).isoformat()

    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get referral earnings summary for {referrer_id_str}: {e}")

    return summary

# في database.py


def process_and_distribute_referral_earnings(now_str: str, days_to_unlock: int) -> int:
    """
    توزيع الأرباح المعلقة من جدول referral_earnings التي مضى عليها days_to_unlock يوم
    بناءً على صافي إيداع المدعو بعد أي سحب.
    """
    updated_count = 0

    try:
        from datetime import datetime, timedelta
        import sqlite3, json, time
        from config import get_config

        config = get_config()
        commissions = config.get("REFERRAL_COMMISSION", {"1": 0.0, "2": 0.0, "3": 0.0})

        with get_connection() as conn:
            cur = conn.cursor()

            print(f"🔍 جاري توزيع الأرباح المعلقة ({days_to_unlock} يوم)...")

            # حساب تاريخ القطع
            try:
                cutoff_date = datetime.fromisoformat(now_str) - timedelta(days=days_to_unlock) if now_str else datetime.now() - timedelta(days=days_to_unlock)
            except:
                cutoff_date = datetime.now() - timedelta(days=days_to_unlock)

            cutoff_date_str = cutoff_date.isoformat()
            print(f"📅 تاريخ القطع: {cutoff_date_str}")

            # جلب الأرباح المؤهلة للتوزيع
            cur.execute("""
                SELECT id, referrer_id, invited_id, amount, level, created_at
                FROM referral_earnings
                WHERE status = 'pending' AND created_at <= ?
            """, (cutoff_date_str,))

            eligible_earnings = cur.fetchall()
            print(f"📊 عدد الأرباح المؤهلة: {len(eligible_earnings)}")

            if not eligible_earnings:
                print("ℹ️ لا توجد أرباح مؤهلة حسب التاريخ")
                return 0

            for earning in eligible_earnings:
                try:
                    earning_id = earning['id']
                    referrer_id = earning['referrer_id']
                    invited_id = earning['invited_id']
                    level = int(earning['level'])
                    created_at = earning['created_at']

                    # جلب بيانات المدعو
                    cur.execute("SELECT total_deposited, balance, has_withdrawn FROM user_accounts WHERE user_id = ?", (invited_id,))
                    invited_user = cur.fetchone()

                    if not invited_user:
                        print(f"⚠️ المدعو غير موجود: {invited_id}")
                        continue

                    total_deposited = invited_user['total_deposited'] or 0
                    has_withdrawn = invited_user['has_withdrawn'] or 0

                    # صافي الإيداع الفعلي
                    net_deposit = max(total_deposited - has_withdrawn, 0)

                    # حساب نسبة المستوى
                    level_percent = float(commissions.get(str(level), 0)) / 100
                    referral_amount = net_deposit * level_percent

                    if referral_amount <= 0:
                        print(f"⚠️ ربح صفر بسبب صافي إيداع المدعو: {invited_id}")
                        continue

                    print(f"\n💰 توزيع ربح {earning_id}: {referrer_id} يحصل على {referral_amount:.2f} ل.س (مستوى {level})")

                    # تحديث حساب المحيل
                    cur.execute("""
                        UPDATE user_accounts
                        SET balance = balance + ?,
                            referral_pending = 0,
                            referral_available = 0,
                            referral_total = COALESCE(referral_total,0) + ?
                        WHERE user_id = ?
                    """, (referral_amount, referral_amount, referrer_id))

                    # تسجيل المعاملة
                    cur.execute("""
                        INSERT INTO transaction_logs
                        (user_id, type, amount, method, status, timestamp, extra)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    """, (referrer_id, 'referral_payout', referral_amount, 'referral', 'success', time.time(),
                          json.dumps({
                              'earning_id': earning_id,
                              'invited_id': invited_id,
                              'level': level,
                              'net_deposit': net_deposit,
                              'distribution_date': datetime.now().isoformat()
                          })))

                    # تحديث حالة الربح
                    cur.execute("""
                        UPDATE referral_earnings
                        SET status = 'distributed',
                            distributed_at = ?
                        WHERE id = ?
                    """, (datetime.now().isoformat(), earning_id))

                    updated_count += 1

                except Exception as e:
                    print(f"❌ خطأ في معالجة ربح {earning_id}: {e}")
                    import traceback
                    traceback.print_exc()
                    continue

            conn.commit()
            print(f"\n🎯 نتائج التوزيع: {updated_count} أرباح موزعة")

    except Exception as e:
        print(f"❌ خطأ في process_and_distribute_referral_earnings: {e}")
        import traceback
        traceback.print_exc()
        return 0

    return updated_count



def set_system_setting(key: str, value: str):
    """
    تخزين أو تحديث قيمة إعداد نظام معين (Key/Value).
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                INSERT OR REPLACE INTO system_settings (key, value) 
                VALUES (?, ?)
                """,
                (key, value)
            )
            conn.commit()
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to set system setting {key}: {e}")
        
# في database.py

def get_user_transaction_logs(user_id_str: str, status: str = 'success', limit: int = 20) -> list[dict]:
    """
    جلب سجلات المعاملات الخاصة بمستخدم معين وبحالة معينة.
    :param user_id_str: معرف المستخدم.
    :param status: حالة المعاملة ('success', 'pending', إلخ).
    :param limit: الحد الأقصى للسجلات المراد جلبها.
    """
    logs = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                SELECT transaction_id, user_id, type, amount, method, timestamp
                FROM transaction_logs
                WHERE user_id = ? AND status = ?
                ORDER BY timestamp DESC
                LIMIT ?
                """,
                (user_id_str, status, limit)
            )
            rows = cur.fetchall()
            logs = [dict(row) for row in reversed(rows)] # ⬅️ عكس الترتيب ليتوافق مع السلوك الأصلي (عرض الأحدث)
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch user transactions for {user_id_str}: {e}")
        
    return logs
def update_transaction_status(log_id: str, new_status: str) -> dict | None:
    """
    تحديث حالة سجل معاملة في transaction_logs.
    """
    if not log_id or log_id == "None":
        print("❌ log_id فارغ أو None!")
        return None
        
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 🔧 محاولة التحديث باستخدام id أولاً (إذا كان log_id رقماً)
            if log_id.isdigit():
                cur.execute(
                    "UPDATE transaction_logs SET status = ?, timestamp = ? WHERE id = ?",
                    (new_status, int(time.time()), int(log_id))
                )
                if cur.rowcount > 0:
                    print(f"✅ تم تحديث حالة المعاملة (باستخدام id) {log_id} إلى {new_status}")
                else:
                    # 🔧 محاولة باستخدام transaction_id
                    cur.execute(
                        "UPDATE transaction_logs SET status = ?, timestamp = ? WHERE transaction_id = ?",
                        (new_status, int(time.time()), log_id)
                    )
                    if cur.rowcount > 0:
                        print(f"✅ تم تحديث حالة المعاملة (باستخدام transaction_id) {log_id} إلى {new_status}")
            else:
                # 🔧 إذا لم يكن رقماً، فهو transaction_id
                cur.execute(
                    "UPDATE transaction_logs SET status = ?, timestamp = ? WHERE transaction_id = ?",
                    (new_status, int(time.time()), log_id)
                )
                if cur.rowcount > 0:
                    print(f"✅ تم تحديث حالة المعاملة {log_id} إلى {new_status}")
            
            conn.commit()
            
            # 🔧 جلب السجل المحدّث
            if log_id.isdigit():
                cur.execute("SELECT * FROM transaction_logs WHERE id = ?", (int(log_id),))
            else:
                cur.execute("SELECT * FROM transaction_logs WHERE transaction_id = ?", (log_id,))
            
            row = cur.fetchone()
            
            if row:
                result = dict(row)
                
                # 🔧 ضمان أن الحقول الأساسية موجودة
                if 'sent_amount' not in result or result['sent_amount'] is None:
                    result['sent_amount'] = result.get('amount', 0.0)
                
                if 'requested_amount' not in result or result['requested_amount'] is None:
                    result['requested_amount'] = result.get('amount', 0.0)
                
                if not result.get('transaction_id'):
                    result['transaction_id'] = str(result.get('id', ''))
                
                return result
                
            return None
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to update transaction status {log_id}: {e}")
        return None
    
def get_transaction_by_details(client_number: str, amount: float) -> dict | None:
    """
    البحث عن سجل سحب في حالة 'processing' أو 'pending' باستخدام رقم العميل والمبلغ.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 🔧 البحث أولاً باستخدام client_number و amount
            cur.execute("""
                SELECT * FROM transaction_logs 
                WHERE client_number = ? AND ABS(amount - ?) < 1.0
                AND status IN ('processing', 'pending')
                ORDER BY timestamp DESC 
                LIMIT 1
            """, (client_number, amount))
            
            row = cur.fetchone()
            
            if not row:
                # 🔧 محاولة البحث بدون مقارنة دقيقة للمبلغ
                cur.execute("""
                    SELECT * FROM transaction_logs 
                    WHERE client_number = ? AND status IN ('processing', 'pending')
                    ORDER BY timestamp DESC 
                    LIMIT 1
                """, (client_number,))
                row = cur.fetchone()
            
            if row:
                result = dict(row)
                
                # 🔧 ضمان وجود transaction_id
                if not result.get('transaction_id'):
                    result['transaction_id'] = str(result.get('id', ''))
                
                return result
                
            return None
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch transaction by details: {e}")
        return None
# في database.py

def refund_user_balance(user_id_str: str, amount: float) -> bool:
    """
    إعادة مبلغ إلى رصيد المستخدم في حالة فشل السحب.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "UPDATE user_accounts SET balance = balance + ? WHERE user_id = ?",
                (amount, user_id_str)
            )
            conn.commit()
            return cur.rowcount > 0
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to refund balance for {user_id_str}: {e}")
        return False

# جلب معرف المُحيل (هذه وظيفة موجودة مسبقًا أو يتم إنشاؤها)
# في ملف database.py

def get_referrer_id(invited_id: str) -> str | None:
    """جلب معرف المُحيل بناءً على معرف المدعو."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            # ✅ تصحيح: استخدام invited_id بدلاً من referred_id
            cur.execute(
                "SELECT referrer_id FROM referral_relationships WHERE invited_id = ?",
                (invited_id,)
            )
            row = cur.fetchone()
            return row['referrer_id'] if row else None
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get referrer ID: {e}")
        return None
    
    
def get_user_balance(user_id_str: str) -> float:
    """
    جلب الرصيد الحالي للمستخدم من جدول user_accounts.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT balance FROM user_accounts WHERE user_id = ?", 
                (user_id_str,)
            )
            row = cur.fetchone()
            # إرجاع الرصيد (أو 0.0 إذا لم يتم العثور على المستخدم)
            return row['balance'] if row else 0.0
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get balance for {user_id_str}: {e}")
        return 0.0


# ===============================================
# 2. خصم الرصيد (deduct_user_balance)
# ===============================================

def deduct_user_balance(user_id_str: str, amount: float) -> bool:
    """
    خصم مبلغ معين من رصيد المستخدم.
    :return: True إذا تم الخصم بنجاح، False بخلاف ذلك.
    """
    if amount <= 0:
        return False
        
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            # 💡 يتم الخصم شرط أن يكون الرصيد كافيًا (balance >= amount)
            cur.execute(
                "UPDATE user_accounts SET balance = balance - ? WHERE user_id = ? AND balance >= ?",
                (amount, user_id_str, amount)
            )
            conn.commit()
            return cur.rowcount > 0
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to deduct balance for {user_id_str}: {e}")
        return False


# ===============================================
# 3. تسجيل المعاملة (log_transaction_and_get_id)
# ===============================================

def log_transaction_and_get_id(transaction_data: dict) -> str:
    """
    تسجيل معاملة جديدة في transaction_logs.
    :param transaction_data: قاموس يحتوي على تفاصيل المعاملة.
    :return: transaction_id (معرف المعاملة).
    """
    transaction_id = str(uuid.uuid4())
    current_time = int(time.time())
    
    # ضمان وجود الحقول الأساسية
    data = {
        "transaction_id": transaction_id,
        "user_id": transaction_data.get("user_id", "Unknown"),
        "type": transaction_data.get("type", "unknown"), # withdraw/deposit
        "method": transaction_data.get("method", "unknown"), # syriatel/MTN/Bank
        "amount": transaction_data.get("amount", 0.0), # المبلغ المطلوب (قبل الرسوم)
        "requested_amount": transaction_data.get("requested_amount", 0.0),
        "sent_amount": transaction_data.get("sent_amount", 0.0), # المبلغ المرسل فعليًا (بعد الرسوم)
        "fee_percent": transaction_data.get("fee_percent", 0),
        "client_number": transaction_data.get("client_number", None),
        "status": transaction_data.get("status", "pending"),
        "timestamp": current_time
    }
    
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 🔧 التحقق من وجود الأعمدة في الجدول
            cur.execute("PRAGMA table_info(transaction_logs)")
            columns = [col[1] for col in cur.fetchall()]
            
            # بناء استعلام ديناميكي بناءً على الأعمدة المتاحة
            query = "INSERT INTO transaction_logs ("
            values = "VALUES ("
            params = []
            
            for key, value in data.items():
                # استبعاد الحقول غير الموجودة في الجدول
                if key in columns:
                    query += f"{key}, "
                    values += "?, "
                    params.append(value)
            
            query = query.rstrip(", ") + ") "
            values = values.rstrip(", ") + ")"
            final_query = query + values
            
            cur.execute(final_query, params)
            conn.commit()
            return transaction_id
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to log transaction: {e}")
        return ""

# ===============================================
# 4. حفظ طلب سحب يدوي (create_pending_withdrawal)
# ===============================================

def create_pending_withdrawal(user_id: str,currency: str, client_code: str, amount: int, method: str , network: str = '',wallet_address: str = '' , amount_in_usdt: float = 0.0) -> str:
    """
    إنشاء طلب سحب معلّق للمراجعة اليدوية (للمبالغ الكبيرة).
    :return: request_id.
    """
    request_id = str(uuid.uuid4())
    current_time = int(time.time())

    try:
        with get_connection() as conn:
            cur = conn.cursor()
            if 'usdt' in method.lower() and wallet_address:
                effective_client_code = wallet_address
            else:
                effective_client_code = client_code
            cur.execute(
                """
                INSERT INTO pending_withdrawals (
                    request_id, user_id, client_number, requested_amount,original_currency, method, 
                    timestamp, status, network, wallet_address, amount_in_usdt
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                """,
                
                (
                    request_id, user_id, client_code, amount,currency, method, current_time, "pending",network,wallet_address,amount_in_usdt
                )
            )
            conn.commit()
            return request_id
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to create pending withdrawal: {e}")
        return ""
    
def register_referral(invited_id: str, referrer_id: str) -> bool:
    """
    تسجيل علاقة إحالة جديدة.
    ⚠️ لن ينشئ حسابات تلقائياً - المدعو يجب أن يكون له حساب مسبقاً
    """
    # منع الإحالة الذاتية
    if str(invited_id) == str(referrer_id):
        print(f"⚠️ محاولة إحالة ذاتية: {invited_id} -> {referrer_id}")
        return False

    try:
        with DB_LOCK:
            with get_connection() as conn:
                cur = conn.cursor()

                # 1. التحقق من وجود المُحيل
                cur.execute("SELECT 1 FROM user_accounts WHERE user_id = ?", (str(referrer_id),))
                if not cur.fetchone():
                    print(f"⚠️ المُحيل غير موجود: {referrer_id}")
                    return False

                # 2. التحقق من وجود المدعو (بدون إنشاء تلقائي)
                cur.execute("SELECT 1 FROM user_accounts WHERE user_id = ?", (str(invited_id),))
                if not cur.fetchone():
                    print(f"⚠️ المدعو غير موجود: {invited_id}")
                    return False

                # 3. التحقق من عدم وجود علاقة مسبقة
                cur.execute("SELECT 1 FROM referral_relationships WHERE invited_id = ?", (str(invited_id),))
                if cur.fetchone():
                    print(f"⚠️ المدعو مسجل مسبقًا: {invited_id}")
                    return False

                # 4. إدخال العلاقة الجديدة
                current_timestamp = int(time.time())
                cur.execute(
                    """
                    INSERT INTO referral_relationships (invited_id, referrer_id, referral_timestamp) 
                    VALUES (?, ?, ?)
                    """,
                    (str(invited_id), str(referrer_id), current_timestamp)
                )
                conn.commit()
                
                print(f"✅ تم تسجيل إحالة بنجاح: {invited_id} <- {referrer_id}")
                return True

    except sqlite3.Error as e:
        print(f"[SQLite ERROR] register_referral failed for {invited_id} <- {referrer_id}: {e}")
        return False
    
def add_referral_earning(invited_id: str, deposit_amount: float, level: int = 1) -> bool:
    """
    إضافة أرباح إحالة عند إيداع مبلغ من قبل المدعو.
    - invited_id: معرف المُدعو (الذي أودع)
    - deposit_amount: المبلغ المودع (بالليرة السورية)
    - level: مستوى الإحالة (1 = مباشر، 2 = غير مباشر، إلخ)
    """
    from config import get_config
    
    # 1. جلب معرف المُحيل
    referrer_id = get_referrer_id(invited_id)
    if not referrer_id:
        return False

    # 2. جلب نسبة العمولة من الكونفيغ (مستوى 1)
    config = get_config()
    commission_rate = config.get("REFERRAL_COMMISSION", {}).get(str(level), 0.05)  # 5% افتراضيًا
    earning_amount = deposit_amount * commission_rate

    if earning_amount <= 0:
        return False

    # 3. توليد معرف فريد للربح
    earning_id = f"ref_{invited_id}_{referrer_id}_{int(time.time())}"

    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # التحقق من عدم تكرار الربح لنفس العملية
            cur.execute("SELECT 1 FROM referral_earnings WHERE id = ?", (earning_id,))
            if cur.fetchone():
                return False

            # إدخال سجل الربح
            cur.execute("""
                INSERT INTO referral_earnings (id, referrer_id, invited_id, amount, status, level, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """, (
                earning_id,
                referrer_id,
                invited_id,
                earning_amount,
                "pending",
                level,
                datetime.now().isoformat()
            ))
            
            # تحديث ملخص الإحالة في حساب المُحيل
            cur.execute("""
                UPDATE user_accounts
                SET 
                    referral_pending = COALESCE(referral_pending, 0) + ?,
                    referral_total = COALESCE(referral_total, 0) + ?
                WHERE user_id = ?
            """, (earning_amount, earning_amount, referrer_id))
            
            return True

    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to add referral earning: {e}")
        return False
    
# في database.py
def get_all_transaction_logs(status: str = None, limit: int = 100) -> list[dict]:
    """
    جلب جميع سجلات المعاملات (لكل المستخدمين) لأغراض الإدارة.
    """
    logs = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            query = "SELECT * FROM transaction_logs"
            params = []
            if status:
                query += " WHERE status = ?"
                params.append(status)
            query += " ORDER BY timestamp DESC LIMIT ?"
            params.append(limit)
            cur.execute(query, params)
            rows = cur.fetchall()
            logs = [dict(row) for row in rows]
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch all transactions: {e}")
    return logs

# في database.py

def get_system_setting(key: str, default=None):
    """
    جلب قيمة من جدول system_settings.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT value FROM system_settings WHERE key = ?", (key,))
            row = cur.fetchone()
            return row["value"] if row else default
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get system setting '{key}': {e}")
        return default
    
# في database.py
def is_user_active(user_id):
    """
    التحقق مما إذا كان المستخدم نشطاً:
    يعتبر نشطاً إذا قام بإيداع واحد ناجح على الأقل (بدون حساب الصافي).
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # فحص وجود أي إيداع ناجح
            cur.execute("""
                SELECT 1 FROM transaction_logs 
                WHERE user_id = ? 
                AND type = 'deposit' 
                AND status = 'success'
                LIMIT 1
            """, (str(user_id),))
            
            result = cur.fetchone()
            
            if result:
                print(f"✅ المستخدم {user_id} نشط (وجدنا إيداع ناجح)")
                return True
            else:
                print(f"ℹ️ المستخدم {user_id} غير نشط (لا يوجد إيداع ناجح بنوع deposit)")
                return False
                
    except Exception as e:
        print(f"❌ خطأ في التحقق من نشاط المستخدم {user_id}: {e}")
        return False

def get_active_referrals(referrer_id):
    """
    جلب قائمة الإحالات النشطة (حسب is_user_active)
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()

            cur.execute("""
                SELECT rr.invited_id, ua.username
                FROM referral_relationships rr
                LEFT JOIN user_accounts ua ON ua.user_id = rr.invited_id
                WHERE rr.referrer_id = ?
            """, (str(referrer_id),))

            rows = cur.fetchall()
            if not rows:
                return []

            active_refs = []
            for row in rows:
                invited_id = row["invited_id"]
                if is_user_active(invited_id):
                    active_refs.append({
                        "user_id": invited_id,
                        "username": row["username"]
                    })

            return active_refs

    except Exception as e:
        print(f"❌ خطأ في جلب الإحالات النشطة لـ {referrer_id}: {e}")
        return []

def get_active_referrals_count(referrer_id):
    """
    حساب عدد الإحالات النشطة التي قامت بإيداع صافي موجب.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 1. جلب كل من دعاهم هذا الشخص
            cur.execute("""
                SELECT invited_id 
                FROM referral_relationships 
                WHERE referrer_id = ?
            """, (str(referrer_id),))
            
            rows = cur.fetchall()
            if not rows:
                return 0
            
            # 2. فحص كل مستخدم هل هو نشط (لديه رصيد) أم لا
            active_count = 0
            for row in rows:
                # نستخدم الدالة الجاهزة التي تفحص الرصيد من جدول الترانزاكشن
                if is_user_active(row[0]): 
                    active_count += 1
            
            return active_count
            
    except Exception as e:
        print(f"❌ خطأ في حساب عدد الإحالات النشطة لـ {referrer_id}: {e}")
        return 0

def has_minimum_active_referrals(referrer_id, minimum=3):
    """
    التحقق مما إذا كان للمُحيل عدد معين من الإحالات النشطة (يستخدم الدالة أعلاه)
    """
    count = get_active_referrals_count(referrer_id)
    print(f"📊 فحص الأهلية لـ {referrer_id}: لديه {count} إحالة نشطة (المطلوب {minimum})")
    return count >= minimum
  
def deduct_referral_on_loss(user_id: str, prev_balance: float, new_balance: float, conn=None):
    from config import get_config
    withdrawn_amount = prev_balance - new_balance
    if withdrawn_amount <= 0:
        return

    referrer_id = get_referrer_id(user_id)
    if not referrer_id:
        return

    config = get_config()
    loss_deduction_percent = config.get("REFERRAL_LOSS_DEDUCTION_PERCENT", 100)
    if loss_deduction_percent <= 0:
        return

    deduction_amount = withdrawn_amount * (loss_deduction_percent / 100.0)

    try:
        # إذا لم يُمرر conn، افتح اتصال جديد
        if conn is None:
            with DB_LOCK:
                with get_connection() as conn:
                    _deduct(conn, referrer_id, deduction_amount, user_id, withdrawn_amount, loss_deduction_percent)
        else:
            _deduct(conn, referrer_id, deduction_amount, user_id, withdrawn_amount, loss_deduction_percent)
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to deduct referral on loss for {referrer_id}: {e}")


def _deduct(conn, referrer_id, deduction_amount, user_id, withdrawn_amount, percent):
    cur = conn.cursor()
    cur.execute("SELECT 1 FROM user_accounts WHERE user_id = ?", (referrer_id,))
    if not cur.fetchone():
        return

    cur.execute("""
        UPDATE user_accounts
        SET 
            referral_pending = MAX(0, referral_pending - ?),
            referral_total = MAX(0, referral_total - ?)
        WHERE user_id = ?
    """, (deduction_amount, deduction_amount, referrer_id))

    add_admin_history(
        user_identifier=referrer_id,
        operation_type="referral_loss_deduction",
        amount=deduction_amount,
        method="auto_deduct",
        status="success",
        extra={
            "deducted_from_user": user_id,
            "withdrawn_amount": withdrawn_amount,
            "percent": percent
        }
    )

def get_promo_code(code: str, admin_check: bool = False, check_published: bool = True) -> dict | None:
    """
    جلب بيانات كود الهدايا من قاعدة البيانات.
    
    :param code: رمز الكود
    :param admin_check: إذا كان True، يرجع الكود حتى لو uses_left = 0 (للعرض في لوحة الأدمن)
    :param check_published: إذا كان True، يرجع فقط الأكواد المنشورة للمستخدمين العاديين
    :return: بيانات الكود أو None
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            if admin_check:
                # للأدمن: يرجع الكود حتى لو uses_left = 0 و بغض النظر عن published
                cur.execute(
                    "SELECT code, value, max_uses, uses_left, published, extra FROM promo_codes WHERE code = ?", 
                    (code.upper(),)
                )
            else:
                if check_published:
                    # للمستخدمين العاديين: يرجع فقط الأكواد المنشورة والملائمة للاستخدام
                    cur.execute(
                        "SELECT code, value, max_uses, uses_left, published, extra FROM promo_codes WHERE code = ? AND uses_left > 0 AND published = 1", 
                        (code.upper(),)
                    )
                else:
                    # السماح باستخدام أي كود فعال (حتى غير المنشور)
                    cur.execute(
                        "SELECT code, value, max_uses, uses_left, published, extra FROM promo_codes WHERE code = ? AND uses_left > 0", 
                        (code.upper(),)
                    )
            
            row = cur.fetchone()
            if row:
                return dict(row)
            return None
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] get_promo_code('{code}') failed: {e}")
        return None

def get_promo_code_unpublished(code: str) -> dict | None:
    """
    جلب كود البونص حتى لو لم يكن منشوراً (للاستخدام الخاص).
    يستخدم هذا للمستخدمين الذين حصلوا على الكود مباشرة من الأدمن.
    
    :param code: رمز الكود
    :return: بيانات الكود أو None
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT code, value, max_uses, uses_left, published FROM promo_codes WHERE code = ? AND uses_left > 0",
                (code.upper(),)
            )
            row = cur.fetchone()
            if row:
                return dict(row)
            return None
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] get_promo_code_unpublished('{code}') failed: {e}")
        return None
def create_private_promo_code(code: str, value: float, max_uses: int, uses_left: int = None, note: str = "كود خاص"):
    """
    إنشاء كود بونص خاص (غير منشور) يمكن للأدمن مشاركته بشكل خاص.
    
    :param code: رمز الكود
    :param value: قيمة الكود
    :param max_uses: الحد الأقصى للاستخدامات
    :param uses_left: الاستخدامات المتبقية (إذا None: uses_left = max_uses)
    :param note: ملاحظة للكود (تخزن في extra)
    """
    if uses_left is None:
        uses_left = max_uses
    
    extra_data = {
        "note": note,
        "is_private": True,
        "created_at": datetime.now().isoformat()
    }
    
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("""
            INSERT INTO promo_codes (code, value, max_uses, uses_left, published, extra)
            VALUES (?, ?, ?, ?, ?, ?)
            ON CONFLICT(code) DO UPDATE SET
                value = excluded.value,
                max_uses = excluded.max_uses,
                uses_left = excluded.uses_left,
                published = excluded.published,
                extra = excluded.extra
        """, (
            code.upper(), 
            value, 
            max_uses, 
            uses_left, 
            0,  # غير منشور
            json.dumps(extra_data)
        ))
        conn.commit()


def consume_promo_code(code: str, user_id: str = None) -> bool:
    """
    تقليل عدد استخدامات كود الهدايا بواحد، مع التحقق من استخدام المستخدم له سابقاً.
    
    :param code: رمز الكود
    :param user_id: معرف المستخدم (اختياري، للتسجيل)
    :return: True إذا تم الاستهلاك بنجاح، False بخلاف ذلك
    """
    try:
        with DB_LOCK:
            with get_connection() as conn:
                cur = conn.cursor()
                
                # التحقق من وجود الكود ووجود استخدامات متبقية
                cur.execute(
                    "SELECT uses_left FROM promo_codes WHERE code = ? AND uses_left > 0",
                    (code.upper(),)
                )
                row = cur.fetchone()
                if not row:
                    return False
                
                # التحقق مما إذا كان المستخدم استخدم الكود من قبل
                if user_id:
                    cur.execute(
                        "SELECT 1 FROM promo_code_usage WHERE code = ? AND user_id = ?",
                        (code.upper(), user_id)
                    )
                    if cur.fetchone():
                        print(f"⚠️ المستخدم {user_id} حاول إعادة استخدام الكود {code}")
                        return False
                
                # تقليل عدد الاستخدامات
                cur.execute(
                    "UPDATE promo_codes SET uses_left = uses_left - 1 WHERE code = ? AND uses_left > 0",
                    (code.upper(),)
                )
                
                if cur.rowcount == 0:
                    return False
                
                # تسجيل استخدام المستخدم للكود
                if user_id:
                    cur.execute(
                        "INSERT OR IGNORE INTO promo_code_usage (code, user_id) VALUES (?, ?)",
                        (code.upper(), user_id)
                    )
                
                conn.commit()
                return True
                
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] consume_promo_code('{code}') failed: {e}")
        return False
    
def get_all_active_promo_codes() -> list[dict]:
    """جلب جميع أكواد الهدايا النشطة (التي لديها استخدامات متبقية)."""
    codes = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT code, value, max_uses, uses_left FROM promo_codes WHERE uses_left > 0"
            )
            rows = cur.fetchall()
            codes = [dict(row) for row in rows]
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] get_all_active_promo_codes() failed: {e}")
    return codes

def add_promo_code(code: str, value: float, max_uses: int, uses_left: int = None, published: bool = False):
    """
    إضافة كود هدية جديد.
    
    :param code: رمز الكود
    :param value: قيمة الكود
    :param max_uses: الحد الأقصى للاستخدامات
    :param uses_left: الاستخدامات المتبقية (إذا None: uses_left = max_uses)
    :param published: هل الكود منشور للمستخدمين؟
    """
    if uses_left is None:
        uses_left = max_uses  # يكون فعالاً فوراً
    
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("""
            INSERT INTO promo_codes (code, value, max_uses, uses_left, published)
            VALUES (?, ?, ?, ?, ?)
        """, (code.upper(), value, max_uses, uses_left, 1 if published else 0))

def delete_promo_code(code: str):
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("DELETE FROM promo_codes WHERE code = ?", (code.upper(),))

def update_promo_code_value(code: str, new_value: float):
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("UPDATE promo_codes SET value = ? WHERE code = ?", (new_value, code.upper()))

def set_promo_code_uses_left(code: str, uses_left: int):
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("UPDATE promo_codes SET uses_left = ? WHERE code = ?", (uses_left, code.upper()))

# في database.py

def update_promo_code(code: str, new_value: float, new_max_uses: int = None):
    """
    تحديث قيمة كود الهدايا (والحد الأقصى للاستخدامات اختياريًا).
    :param code: رمز الكود
    :param new_value: القيمة الجديدة
    :param new_max_uses: (اختياري) الحد الأقصى الجديد للاستخدامات
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            if new_max_uses is not None:
                cur.execute(
                    "UPDATE promo_codes SET value = ?, max_uses = ?, uses_left = ? WHERE code = ?",
                    (new_value, new_max_uses, new_max_uses, code.upper())
                )
            else:
                cur.execute(
                    "UPDATE promo_codes SET value = ? WHERE code = ?",
                    (new_value, code.upper())
                )
            return cur.rowcount > 0
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to update promo code '{code}': {e}")
        return False

def get_all_promo_codes() -> dict[str, dict]:
    """
    جلب جميع أكواد الهدايا (حتى تلك ذات uses_left = 0) كقاموس.
    العودة: {code: {code, value, max_uses, uses_left}}
    """
    codes = {}
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT code, value, max_uses, uses_left FROM promo_codes")
            rows = cur.fetchall()
            for row in rows:
                codes[row["code"]] = dict(row)
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch all promo codes: {e}")
    return codes

# في database.py

def get_all_user_ids() -> list[str]:
    """
    جلب قائمة بجميع معرفات المستخدمين من جدول user_accounts.
    """
    user_ids = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT user_id FROM user_accounts")
            rows = cur.fetchall()
            user_ids = [row["user_id"] for row in rows]
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to fetch all user IDs: {e}")
    return user_ids

# في database.py

def set_config_value(key: str, value):
    """
    دالة متوافقة مع التسمية القديمة: تحفظ قيمة في جدول app_settings.
    """
    _set_setting(key, value)


def get_user_referral_earnings(referrer_id: str) -> list[dict]:
    """
    جلب جميع سجلات أرباح الإحالة لمُحيل معين من جدول referral_earnings.
    :param referrer_id: معرف المستخدم (المُحيل)
    :return: قائمة من القواميس التي تمثل كل سجل ربح
    """
    earnings = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                SELECT id, referrer_id, invited_id, amount, status, level, created_at
                FROM referral_earnings
                WHERE referrer_id = ?
                """,
                (referrer_id,)
            )
            rows = cur.fetchall()
            for row in rows:
                earnings.append(dict(row))
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] get_user_referral_earnings({referrer_id}) failed: {e}")
    return earnings

def delete_referral_earning(invited_id: str, referrer_id: str, level: int):
    """حذف سجل ربح معين بناءً على invited_id, referrer_id, و level."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                DELETE FROM referral_earnings
                WHERE invited_id = ? AND referrer_id = ? AND level = ?
                """,
                (invited_id, referrer_id, level)
            )
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] delete_referral_earning failed: {e}")

def insert_referral_earning(earning_data: dict):
    """إدخال سجل ربح جديد في جدول referral_earnings."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                INSERT INTO referral_earnings (id, referrer_id, invited_id, amount, status, level, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                """,
                (
                    earning_data["earning_id"],
                    earning_data["referrer_id"],
                    earning_data["invited_id"],
                    earning_data["amount"],
                    earning_data["status"],
                    earning_data["level"],
                    earning_data["created_at"]
                )
            )
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] insert_referral_earning failed: {e}")


def update_user_referral_summary(user_id: str, pending: float, available: float, total: float):
    """تحديث حقول ملخص الإحالة في جدول user_accounts."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                """
                UPDATE user_accounts
                SET referral_pending = ?, referral_available = ?, referral_total = ?
                WHERE user_id = ?
                """,
                (pending, available, total, user_id)
            )
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] update_user_referral_summary failed: {e}")


def is_user_banned(user_id: str) -> bool:
    """التحقق مما إذا كان المستخدم محظورًا."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT 1 FROM banned_users WHERE user_id = ?", (user_id,))
            return cur.fetchone() is not None
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to check if user {user_id} is banned: {e}")
        return False

def ban_user(user_id: str) -> bool:
    """حظر مستخدم."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("INSERT OR IGNORE INTO banned_users (user_id) VALUES (?)", (user_id,))
            return cur.rowcount > 0
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to ban user {user_id}: {e}")
        return False

def unban_user(user_id: str) -> bool:
    """رفع الحظر عن مستخدم."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("DELETE FROM banned_users WHERE user_id = ?", (user_id,))
            return cur.rowcount > 0
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to unban user {user_id}: {e}")
        return False
    

def get_all_user_accounts() -> dict[str, dict]:
    """
    جلب جميع حسابات المستخدمين من جدول user_accounts.
    العودة: قاموس {user_id: {user_data}}
    """
    users = {}
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT * FROM user_accounts")
            rows = cur.fetchall()
            for row in rows:
                user_id = str(row['user_id'])
                user_data = dict(row)
                user_data['user_history'] = json.loads(user_data.get('user_history', '[]'))
                user_data['has_withdrawn'] = bool(user_data.get('has_withdrawn', 0))
                users[user_id] = user_data
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get all user accounts: {e}")
    return users


def get_transaction_logs(limit: int = None) -> list[dict]:
    """
    جلب جميع سجلات المعاملات من جدول transaction_logs.
    :param limit: الحد الأقصى للسجلات المراد جلبها (اختياري)
    :return: قائمة من القواميس التي تمثل سجلات المعاملات
    """
    logs = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            query = "SELECT * FROM transaction_logs ORDER BY timestamp DESC"
            if limit:
                query += f" LIMIT {limit}"
            cur.execute(query)
            rows = cur.fetchall()
            logs = [dict(row) for row in rows]
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get transaction logs: {e}")
    return logs


def is_deposit_withdraw_enabled() -> bool:
    """التحقق مما إذا كانت خاصية الشحن والسحب مفعلة."""
    enabled = _get_setting("deposit_withdraw_enabled", True)
    if isinstance(enabled, str):
        return enabled.lower() == 'true'
    return bool(enabled)

# أضف هذه الدوال إلى database.py

def get_pending_withdrawals():
    """جلب جميع طلبات السحب المعلقة."""
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("SELECT * FROM pending_withdrawals WHERE status = 'pending'")
        rows = cur.fetchall()
        return [dict(row) for row in rows]


def update_pending_withdrawal_status(request_id: str, status: str):
    """تحديث حالة طلب سحب معلق."""
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("""
            UPDATE pending_withdrawals 
            SET status = ? 
            WHERE request_id = ?
        """, (status, request_id))
        conn.commit()


def delete_pending_withdrawal(request_id: str):
    """حذف طلب سحب معلق."""
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute("DELETE FROM pending_withdrawals WHERE request_id = ?", (request_id,))
        conn.commit()

def update_promo_code_published_status(code: str, published: bool):
    """
    تغيير حالة النشر لكود البونص.
    
    :param code: رمز الكود
    :param published: True للنشر، False لإخفائه
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "UPDATE promo_codes SET published = ? WHERE code = ?",
                (1 if published else 0, code.upper())
            )
            return cur.rowcount > 0
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to update promo code publish status '{code}': {e}")
        return False
    

def has_user_used_promo_code(user_id: str, code: str) -> bool:
    """
    التحقق مما إذا كان المستخدم قد استخدم هذا الكود من قبل.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT 1 FROM promo_code_usage WHERE code = ? AND user_id = ?",
                (code.upper(), user_id)
            )
            return cur.fetchone() is not None
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to check promo code usage: {e}")
        return True  # في حالة الخطأ، نفترض أنه استخدمه لمنع الاستغلال


def record_promo_code_usage(user_id: str, code: str) -> bool:
    """
    تسجيل استخدام المستخدم للكود.
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "INSERT OR IGNORE INTO promo_code_usage (code, user_id) VALUES (?, ?)",
                (code.upper(), user_id)
            )
            return True
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to record promo code usage: {e}")
        return False


def get_promo_code_users(code: str) -> list[str]:
    """
    جلب قائمة المستخدمين الذين استخدموا كود معين.
    """
    users = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT user_id FROM promo_code_usage WHERE code = ?",
                (code.upper(),)
            )
            rows = cur.fetchall()
            users = [row['user_id'] for row in rows]
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get promo code users: {e}")
    return users


def get_transaction_by_id(log_id: str) -> dict | None:
    """
    جلب سجل معاملة بواسطة log_id (يمكن أن يكون id أو transaction_id).
    
    :param log_id: معرف السجل (رقم أو uuid)
    :return: بيانات السجل أو None
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # محاولة البحث باستخدام id (إذا كان رقماً)
            if log_id.isdigit():
                cur.execute("SELECT * FROM transaction_logs WHERE id = ?", (int(log_id),))
                row = cur.fetchone()
                if row:
                    return dict(row)
            
            # محاولة البحث باستخدام transaction_id
            cur.execute("SELECT * FROM transaction_logs WHERE transaction_id = ?", (log_id,))
            row = cur.fetchone()
            if row:
                return dict(row)
            
            # البحث في pending_withdrawals إذا لم يتم العثور في transaction_logs
            cur.execute("SELECT * FROM pending_withdrawals WHERE request_id = ?", (log_id,))
            row = cur.fetchone()
            if row:
                return dict(row)
            
            return None
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] Failed to get transaction by id {log_id}: {e}")
        return None
    
def get_correct_refund_amount(transaction_id: str) -> tuple[float, float]:
    """
    جلب المبلغ الصحيح للاسترداد (قبل الخصم) مع حساب الرسوم.
    
    :return: (amount_before_fee, fee_amount)
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()            
            # البحث عن المعاملة
            cur.execute("""
                SELECT requested_amount, sent_amount, amount, fee_percent 
                FROM transaction_logs 
                WHERE transaction_id = ? OR id = ?
            """, (transaction_id, transaction_id))
            
            row = cur.fetchone()
            
            if row:
                requested_amount = row['requested_amount'] or 0.0
                sent_amount = row['sent_amount'] or 0.0
                amount = row['amount'] or 0.0
                fee_percent = row['fee_percent'] or 0.0
                
                # الأفضلية: requested_amount > amount > sent_amount
                amount_before_fee = requested_amount if requested_amount > 0 else amount
                
                # إذا كان amount_before_fee = sent_amount (معنى ذلك أن المبلغ بعد الخصم)
                if amount_before_fee <= sent_amount:
                    # حساب المبلغ قبل الخصم بناءً على نسبة الرسوم
                    if fee_percent > 0:
                        amount_before_fee = sent_amount / (1 - (fee_percent / 100.0))
                    else:
                        amount_before_fee = sent_amount
                
                fee_amount = amount_before_fee - sent_amount
                
                print(f"🔍 استرداد: قبل الخصم={amount_before_fee:,.0f}, بعد الخصم={sent_amount:,.0f}, رسوم={fee_amount:,.0f} ({fee_percent}%)")
                return float(amount_before_fee), float(fee_amount)
            
            return float(amount), 0.0
            
    except Exception as e:
        print(f"[SQLite ERROR] Failed to get correct refund amount: {e}")
        return float(amount), 0.0

def get_transaction_by_request_id(request_id):
    with get_connection() as conn:
        cur = conn.cursor()
        cur.execute(
            "SELECT * FROM transaction_logs WHERE request_id = ?",
            (request_id,)
        )
        return cur.fetchone()

def get_user_referral_summary(user_id):
    """
    ترجع ملخص الإحالة للمستخدم من جدول user_referral_summary.
    إذا لم يوجد سجل، ترجع القيم الافتراضية.
    """
    conn = sqlite3.connect("users.db")
    cur = conn.cursor()
    cur.execute("""
        SELECT referral_pending, referral_available, referral_total, last_processed_deposit_id
        FROM user_referral_summary
        WHERE user_id = ?
    """, (user_id,))
    row = cur.fetchone()
    conn.close()

    if row:
        return {
            "referral_pending": row[0],
            "referral_available": row[1],
            "referral_total": row[2],
            "last_processed_deposit_id": row[3]
        }
    else:
        return {
            "referral_pending": 0.0,
            "referral_available": 0.0,
            "referral_total": 0.0,
            "last_processed_deposit_id": None
        }
        
def update_user_referral_summary(user_id, pending, available, total, last_processed_deposit_id=None):
    """
    تحدث ملخص الإحالة للمستخدم وتخزن آخر إيداع تمت معالجته.
    إذا لم يوجد سجل سابق، يتم إنشاؤه.
    """
    conn = sqlite3.connect("users.db")
    cur = conn.cursor()

    # إدراج أو تحديث السجل
    cur.execute("""
        INSERT INTO user_referral_summary (user_id, referral_pending, referral_available, referral_total, last_processed_deposit_id)
        VALUES (?, ?, ?, ?, ?)
        ON CONFLICT(user_id) DO UPDATE SET
            referral_pending=excluded.referral_pending,
            referral_available=excluded.referral_available,
            referral_total=excluded.referral_total,
            last_processed_deposit_id=excluded.last_processed_deposit_id
    """, (user_id, pending, available, total, last_processed_deposit_id))

    conn.commit()
    conn.close()


def add_referral_timestamp_column():
    """
    إضافة حقل referral_timestamp لجدول referral_relationships
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # محاولة إضافة الحقل
            cur.execute("""
                ALTER TABLE referral_relationships 
                ADD COLUMN referral_timestamp INTEGER DEFAULT 0
            """)
            
            # تحديث البيانات القديمة
            cur.execute("""
                UPDATE referral_relationships 
                SET referral_timestamp = strftime('%s', 'now')
                WHERE referral_timestamp = 0 OR referral_timestamp IS NULL
            """)
            
            conn.commit()
            print("✅ تم إضافة حقل referral_timestamp وتحديث البيانات")
            return True
            
    except sqlite3.OperationalError as e:
        if "duplicate column" in str(e):
            print("⚠️ الحقل موجود بالفعل")
            return True
        else:
            print(f"❌ خطأ: {e}")
            return False
    except Exception as e:
        print(f"❌ خطأ غير متوقع: {e}")
        return False
def add_referral_timestamp_column():
    """
    إضافة حقل referral_timestamp لجدول referral_relationships
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # محاولة إضافة الحقل
            cur.execute("""
                ALTER TABLE referral_relationships 
                ADD COLUMN referral_timestamp INTEGER DEFAULT 0
            """)
            
            # تحديث البيانات القديمة
            cur.execute("""
                UPDATE referral_relationships 
                SET referral_timestamp = strftime('%s', 'now')
                WHERE referral_timestamp = 0 OR referral_timestamp IS NULL
            """)
            
            conn.commit()
            print("✅ تم إضافة حقل referral_timestamp وتحديث البيانات")
            return True
            
    except sqlite3.OperationalError as e:
        if "duplicate column" in str(e):
            print("⚠️ الحقل موجود بالفعل")
            return True
        else:
            print(f"❌ خطأ: {e}")
            return False
    except Exception as e:
        print(f"❌ خطأ غير متوقع: {e}")
        return False
def get_last_user_deposit(user_id):
    try:
        conn = sqlite3.connect("users.db")
        cur = conn.cursor()
        
        # نستخدم COALESCE لاختيار transaction_id وإذا كان فارغاً نأخذ رقم الـ id التلقائي
        cur.execute("""
            SELECT 
                CASE 
                    WHEN transaction_id IS NOT NULL AND transaction_id != '' THEN transaction_id 
                    ELSE CAST(id AS TEXT) 
                END as unique_tx_id,
                amount, 
                timestamp
            FROM transaction_logs
            WHERE CAST(user_id AS TEXT) = CAST(? AS TEXT) 
              AND UPPER(type) = 'DEPOSIT' 
              AND UPPER(status) = 'SUCCESS'
            ORDER BY timestamp DESC
            LIMIT 1
        """, (str(user_id),))
        
        row = cur.fetchone()
        conn.close()

        if row:
            return {
                "transaction_id": row[0], # سيعود برقم الـ id (مثل "53") إذا كان الترانزاكشن فارغاً
                "amount": row[1],
                "timestamp": row[2]
            }
        return None
    except Exception as e:
        print(f"❌ خطأ في get_last_user_deposit: {e}")
        return None
def get_new_referrals_last_24h(referrer_id: str) -> int:
    try:
        with get_connection() as conn:
            cur = conn.cursor()

            
            # الوقت الحالي بالثواني
            now = int(time.time())
            # الوقت قبل 24 ساعة من الآن تماماً
            twenty_four_hours_ago = now - (24 * 3600)
            print(f"DEBUG: Current Time: {now}")
            print(f"DEBUG: Threshold: {twenty_four_hours_ago}")
            cur.execute('''
                SELECT COUNT(*) FROM referral_relationships 
                WHERE referrer_id = ? AND referral_timestamp >= ?
            ''', (str(referrer_id), twenty_four_hours_ago))
            
            result = cur.fetchone()
            count = result[0] if result else 0
            
            print(f"🔄 فحص آخر 24 ساعة لـ {referrer_id}: وجدنا {count} إحالات")
            return count
    except Exception as e:
        print(f"❌ خطأ: {e}")
        return 0

def give_spins_for_referrals(min_referrals=5, spins_per_user=1):
    """
    تعطي لفة عجلة لكل مستخدم لديه عدد إحالات >= min_referrals
    
    Args:
        min_referrals: الحد الأدنى من الإحالات (default: 5)
        spins_per_user: عدد اللفات لكل مستخدم (default: 1)
    """
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            print(f"🔄 جاري البحث عن مستخدمين لديهم {min_referrals}+ إحالات...")
            
            # 1. البحث عن المستخدمين الذين لديهم 5+ إحالات
            cur.execute('''
                SELECT referrer_id, COUNT(*) as referral_count 
                FROM referral_relationships 
                GROUP BY referrer_id 
                HAVING COUNT(*) >= ?
                ORDER BY referral_count DESC
            ''', (min_referrals,))
            
            eligible_users = cur.fetchall()
            total_eligible = len(eligible_users)
            
            if total_eligible == 0:
                print(f"✅ لا يوجد مستخدمين لديهم {min_referrals}+ إحالات")
                return 0
            
            print(f"✅ تم العثور على {total_eligible} مستخدم")
            
            # 2. إنشاء جدول لتسجيل المكافآت (إذا لم يكن موجوداً)
            cur.execute('''
                CREATE TABLE IF NOT EXISTS referral_spin_rewards (
                    user_id TEXT PRIMARY KEY,
                    referral_count INTEGER DEFAULT 0,
                    spins_given INTEGER DEFAULT 0,
                    rewarded_at INTEGER DEFAULT 0
                )
            ''')
            
            # 3. منح اللفات للمستخدمين الجدد فقط
            users_given_spins = 0
            
            for user_id, referral_count in eligible_users:
                try:
                    # التحقق إذا كان المستخدم قد حصل على المكافأة مسبقاً
                    cur.execute('''
                        SELECT 1 FROM referral_spin_rewards 
                        WHERE user_id = ?
                    ''', (user_id,))
                    
                    already_rewarded = cur.fetchone()
                    
                    if not already_rewarded:
                        # التحقق من وجود حساب المستخدم
                        cur.execute("SELECT 1 FROM user_accounts WHERE user_id = ?", (user_id,))
                        if cur.fetchone():
                            
                            # تخفيض wheel_spins_used (كلما قل الرقم زادت اللفات المتاحة)
                            cur.execute('''
                                UPDATE user_accounts 
                                SET wheel_spins_used = wheel_spins_used - ?
                                WHERE user_id = ?
                            ''', (spins_per_user, user_id))
                            
                            # تسجيل في جدول المكافآت
                            current_time = int(time.time())
                            cur.execute('''
                                INSERT INTO referral_spin_rewards 
                                (user_id, referral_count, spins_given, rewarded_at)
                                VALUES (?, ?, ?, ?)
                            ''', (user_id, referral_count, spins_per_user, current_time))
                            
                            # إضافة إلى user_history بشكل آمن
                            try:
                                import json
                                cur.execute("SELECT user_history FROM user_accounts WHERE user_id = ?", (user_id,))
                                result = cur.fetchone()
                                
                                if result:
                                    current_history = result[0] or '[]'
                                    try:
                                        history_list = json.loads(current_history)
                                    except:
                                        history_list = []
                                    
                                    new_entry = {
                                        "type": "referral_bonus",
                                        "message": f"🎡 حصلت على {spins_per_user} لفة عجلة مجانية لامتلاكك {referral_count} إحالة",
                                        "timestamp": current_time,
                                        "referral_count": referral_count
                                    }
                                    history_list.append(new_entry)
                                    
                                    new_history = json.dumps(history_list, ensure_ascii=False)
                                    
                                    cur.execute('''
                                        UPDATE user_accounts 
                                        SET user_history = ?
                                        WHERE user_id = ?
                                    ''', (new_history, user_id))
                            except Exception as history_error:
                                print(f"   ⚠️ خطأ في تحديث التاريخ للمستخدم {user_id}: {history_error}")
                            
                            users_given_spins += 1
                            print(f"   🎁 {user_id}: {referral_count} إحالة ← 🎡 +{spins_per_user} لفة")
                        else:
                            print(f"   ⚠️ {user_id}: حساب غير موجود (تم تخطيه)")
                    else:
                        print(f"   ⚠️ {user_id}: حصل على المكافأة مسبقاً (تم تخطيه)")
                
                except Exception as e:
                    print(f"   ❌ خطأ في معالجة المستخدم {user_id}: {e}")
                    import traceback
                    traceback.print_exc()
                    continue
            
            conn.commit()
            print(f"\n🎉 تم منح {spins_per_user} لفة لـ {users_given_spins} مستخدم")
            
            # 4. عرض إحصائية
            if users_given_spins > 0:
                print(f"\n📊 الإحصائيات:")
                for user_id, referral_count in eligible_users[:10]:  # أول 10 فقط
                    cur.execute('''
                        SELECT spins_given FROM referral_spin_rewards 
                        WHERE user_id = ?
                    ''', (user_id,))
                    reward_data = cur.fetchone()
                    status = "✅ مكافئ" if reward_data else "❌ لم يكافئ بعد"
                    print(f"   {user_id}: {referral_count} إحالة - {status}")
                
                if len(eligible_users) > 10:
                    print(f"   ... و {len(eligible_users) - 10} مستخدم آخر")
            
            return users_given_spins
            
    except sqlite3.Error as e:
        print(f"[SQLite ERROR] give_spins_for_referrals failed: {e}")
        import traceback
        traceback.print_exc()
        return 0
    

def toggle_deposit_method(method_key: str):
    """
    method_key مثال:
    deposit_usdt
    deposit_cham_cash
    deposit_syriatel_cash
    deposit_hawala
    """
    setting_key = f"{method_key}_enabled"
    current = _get_setting(setting_key, True)
    _set_setting(setting_key, not current)
    return not current


# pending_deposits table fields: id, user_id, amount, operation_code, method, created_at
def create_pending_deposit(user_id: str, operation_code: str, amount: float, method: str) -> int:
    """إنشاء طلب إيداع معلّق وإرجاع معرفه (id)."""
    import sqlite3
    from database import get_connection, DB_LOCK

    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            try:
                cur.execute(
                    "INSERT INTO pending_deposits (user_id, amount, operation_code, method) VALUES (?, ?, ?, ?)",
                    (user_id, amount, operation_code, method)
                )
                return cur.lastrowid
            except sqlite3.IntegrityError as e:
                print(f"❌ خطأ عند إنشاء الطلب: {e}")
                return None



def get_pending_deposit(operation_code: str):
    """جلب طلب إيداع معلّق واحد عبر كود العملية."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT id, user_id, amount, operation_code, method
                FROM pending_deposits
                WHERE operation_code = ?
                LIMIT 1
            """, (operation_code,))
            row = cur.fetchone()
            return dict(row) if row else None


def get_all_pending_deposits():
    """جلب جميع طلبات الإيداع المعلقة."""
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT id, user_id, amount, operation_code, method, created_at
                FROM pending_deposits
                ORDER BY id DESC
            """)
            rows = cur.fetchall()
            return [dict(r) for r in rows]

def get_pending_deposit_by_code(operation_code: str):
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT id, user_id, amount, operation_code, method
                FROM pending_deposits
                WHERE operation_code = ?
                LIMIT 1
            """, (operation_code,))
            row = cur.fetchone()
            return dict(row) if row else None

        
def delete_pending_deposit(operation_code: str):
    with DB_LOCK:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("DELETE FROM pending_deposits WHERE operation_code = ?", (operation_code,))

            
