from telegram import InlineKeyboardButton, InlineKeyboardMarkup, Update
from telegram.constants import ParseMode
from telegram.ext import CallbackContext, ContextTypes, ConversationHandler  
from ishansy_controller import IshansyController
from datetime import datetime, timedelta
import openpyxl
from openpyxl.utils import get_column_letter
import os
import tempfile
import time 
import random 
import json
from database import get_connection , is_user_active
# ====================================================================================================
#                                     دالة مساعدة لتحويل التاريخ (بدون تغيير)
# ====================================================================================================

def get_tx_time(tx: dict) -> datetime | None:
    """محاولة تحويل قيمة 'timestamp' من تنسيق يونكس (int/float) أو تنسيق نصي إلى كائن datetime."""
    try:
        timestamp_value = tx.get("timestamp")
        
        if timestamp_value is None:
            return None
            
        # 1. يونكس (float/int)
        if isinstance(timestamp_value, (float, int)):
            return datetime.fromtimestamp(timestamp_value)
            
        # 2. نص (String)
        elif isinstance(timestamp_value, str):
            try:
                return datetime.strptime(timestamp_value, "%Y-%m-%d %H:%M:%S.%f")
            except ValueError:
                return datetime.strptime(timestamp_value, "%Y-%m-%d %H:%M:%S")
        else:
            return None
    except (KeyError, TypeError, ValueError, OSError) as e:
        # ملاحظة: fromtimestamp قد يرمي OSError إذا كان الرقم كبير جدًا/صغير جدًا
        print(f"⚠️ خطأ في تحويل الوقت: {e} | tx={tx}")
        return None

# ====================================================================================================
#                                     دوال الإحصائيات (Admin Stats) (بدون تغيير)
# ====================================================================================================


async def show_admin_stats(update: Update, context: CallbackContext):
    query = update.callback_query
    await query.answer()

    # ==========================
    #  رصيد الكاشير
    # ==========================
    ishansy = IshansyController("luffy@gmail.com", "Aa@723723")
    cashier_balance = 0

    try:
        if not ishansy.session_cookies:
            if not ishansy.resolve_captcha_and_extract_cookies():
                cashier_balance = 0
        if ishansy.session_cookies:
            cashier_balance = ishansy.get_cashier_balance()

    except:
        cashier_balance = 0

    # ==========================
    #  تحميل البيانات من SQLite
    # ==========================
    from database import get_connection

    with get_connection() as conn:
        cur = conn.cursor()

        # --- users ---
        cur.execute("SELECT * FROM user_accounts")
        user_accounts = cur.fetchall()

        # --- logs ---
        cur.execute("SELECT * FROM transaction_logs")
        transaction_logs = cur.fetchall()

        # --- referral earnings ---
        cur.execute("SELECT * FROM referral_earnings")
        referral_earnings = cur.fetchall()
        
        cur.execute("SELECT requested_amount FROM pending_withdrawals")
        pending_rows = cur.fetchall()

    # ==========================
    #  الإحصائيات الأساسية
    # ==========================
    total_users = len(user_accounts)
    total_deposits = sum(row["total_deposited"] for row in user_accounts)
    total_balance = sum(row["balance"] for row in user_accounts)
    total_bonus = sum(row["bonus_balance"] for row in user_accounts)
    total_usdt = sum(row["usdt_balance"] for row in user_accounts)

    # ==========================
    #  السحوبات
    # ==========================
    manual_withdrawals = 0
    auto_withdrawals = 0
    pending_withdrawals = sum(float(row["requested_amount"]) for row in pending_rows)

    for tx in transaction_logs:
        if tx["type"] != "withdraw":
            continue

        amount = float(tx["amount"])
        status = tx["status"]
        method = tx["method"]

        if status == "success":
            if method == "manual_withdraw":
                manual_withdrawals += amount
            else:
                auto_withdrawals += amount
        

    total_withdrawals = manual_withdrawals + auto_withdrawals

    # ==========================
    # أرباح الإحالات
    # ==========================
    total_referral_earnings = sum(
        row["amount"]
        for row in referral_earnings
        if row["status"] in ("pending", "distributed")
    )

    # ==========================
    # أرباح الألعاب
    # ==========================
    total_wheel_wins = sum(
        row["amount"]
        for row in transaction_logs
        if row["type"] == "wheel_win" and row["status"] == "success"
    )

    total_box_wins = sum(
        row["amount"]
        for row in transaction_logs
        if row["type"] == "box_win" and row["status"] == "success"
    )

    # ==========================
    # أرباح الأكواد
    # ==========================
    total_promo_wins = 0
    for user in user_accounts:
        history = json.loads(user["user_history"]) if user["user_history"] else []
        for tx in history:
            if tx["type"] == "bonus" and tx["method"] == "promo":
                total_promo_wins += float(tx["amount"])

    # ==========================
    # الإحصائيات الزمنية
    # ==========================
    now = datetime.now()
    start_of_day = datetime(now.year, now.month, now.day)
    end_of_day = start_of_day + timedelta(days=1)
    start_of_week = start_of_day - timedelta(days=start_of_day.weekday())
    end_of_week = start_of_week + timedelta(days=7)

    daily_deposits = daily_withdrawals = 0
    weekly_deposits = weekly_withdrawals = 0

    def get_tx_time(tx):
        try:
            return datetime.fromtimestamp(tx["timestamp"])
        except:
            return None

    for tx in transaction_logs:
        if tx["status"] != "success":
            continue
        tx_time = get_tx_time(tx)
        if not tx_time:
            continue

        amount = float(tx["amount"])
        typ = tx["type"]

        if start_of_day <= tx_time < end_of_day:
            if typ == "deposit": daily_deposits += amount
            elif typ == "withdraw": daily_withdrawals += amount

        if start_of_week <= tx_time < end_of_week:
            if typ == "deposit": weekly_deposits += amount
            elif typ == "withdraw": weekly_withdrawals += amount

    # ==========================
    #   إرسال الإحصائيات
    # ==========================
    stats_text = (
        f"📊 *إحصائيات البوت الشاملة* 📊\n\n"
        f"👥 عدد المستخدمين: {total_users}\n"
        f"💼 رصيد الكاشير: {cashier_balance:,.0f} NSP\n\n"
        f"💰 مجموع الإيداعات: {total_deposits:,.0f} ل.س\n"
        f"💸 السحوبات الآلية: {auto_withdrawals:,.0f} ل.س\n"
        f"🧑‍💼 السحوبات اليدوية: {manual_withdrawals:,.0f} ل.س\n"
        f"⏳ السحوبات المعلقة: {pending_withdrawals:,.0f} ل.س\n"
        f"🏦 مجموع أرصدة المستخدمين: {total_balance:,.0f} ل.س\n"
        f"🎁 مجموع البونصات: {total_bonus:,.0f} ل.س\n"
        f"💵 مجموع USDT: {total_usdt:,.2f} USDT\n"
        f"🤝 أرباح الإحالات: {total_referral_earnings:,.0f} ل.س\n"
        f"🎡 أرباح العجلة: {total_wheel_wins:,.0f} ل.س\n"
        f"📦 أرباح الصناديق: {total_box_wins:,.0f} ل.س\n"
        f"🎉 أرباح الأكواد: {total_promo_wins:,.0f} ل.س\n\n"
        f"📆 *اليوم*: {daily_deposits:,.0f} إيداع / {daily_withdrawals:,.0f} سحب\n"
        f"📅 *الأسبوع*: {weekly_deposits:,.0f} إيداع / {weekly_withdrawals:,.0f} سحب\n"
    )

    keyboard = [
        [InlineKeyboardButton("📥 تصدير Excel", callback_data="export_weekly_excel")],
        [InlineKeyboardButton("🔙 العودة للوحة الأدمن", callback_data="back_to_admin_panel")]
    ]

    await query.message.reply_text(
        stats_text,
        reply_markup=InlineKeyboardMarkup(keyboard),
        parse_mode="Markdown"
    )

# ----------------------------------------------------------------------------------------------------
#                                     دالة تصدير الإكسل (Excel Export) - تم التعديل
# ----------------------------------------------------------------------------------------------------
async def export_weekly_excel(update: Update, context: CallbackContext):
    from database import get_all_user_accounts, get_transaction_logs
    query = update.callback_query
    await query.answer()

    today = datetime.now()
    start_of_day = datetime(today.year, today.month, today.day)
    end_of_day = start_of_day + timedelta(days=1)
    start_of_week = start_of_day - timedelta(days=start_of_day.weekday())
    end_of_week = start_of_week + timedelta(days=7)

    # 🔹 جلب البيانات من SQLite
    user_accounts = get_all_user_accounts()           # dict {user_id: {...}}
    transaction_logs = get_transaction_logs()         # list of dicts

    # 🔹 خريطة أسماء المستخدمين
    user_names = {
        str(user_id): data.get("username", f"user_{user_id}")
        for user_id, data in user_accounts.items()
    }

    weekly_rows = []
    daily_rows = []
    total_rows = []

    # 🔹 نوع العملية (عرض عربي)
    tx_type_map = {
        "deposit": "إيداع",
        "withdraw": "سحب",
        "wheel_win": "ربح عجلة",
        "box_win": "ربح صندوق",
        "promo_code_win": "كود هدية",
        "bonus": "بونص ترحيبي",
        "referral_bonus": "إحالة",
    }

    for tx in transaction_logs:
        # ✅ فقط العمليات الناجحة
        if tx.get("status") != "success":
            continue

        tx_time = get_tx_time(tx)
        if tx_time is None:
            continue


        user_id = tx.get("user_id")
        if not user_id:
            continue

        username = user_names.get(str(user_id), str(user_id))
        tx_type = tx_type_map.get(tx.get("type"), tx.get("type", "أخرى"))
        amount = float(tx.get("amount", 0.0))
        method = tx.get("method", "N/A")
        status = tx.get("status", "unknown")

        row = [
            str(user_id),
            username,
            tx_type,
            amount,
            method,
            status,
            tx_time.strftime("%Y-%m-%d %H:%M:%S")
        ]

        if start_of_week <= tx_time < end_of_week:
            weekly_rows.append(row)
        if start_of_day <= tx_time < end_of_day:
            daily_rows.append(row)

        total_rows.append(row)

    if not total_rows:
        await query.message.reply_text("⚠️ لا توجد معاملات لتصديرها.")
        return

    # --- إنشاء ملف Excel ---
    wb = openpyxl.Workbook()
    ws_weekly = wb.active
    ws_weekly.title = "الأسبوعية"

    headers = ["معرف المستخدم", "اسم المستخدم", "نوع العملية", "المبلغ", "الطريقة", "الحالة", "الوقت"]
    ws_weekly.append(headers)

    for row in weekly_rows:
        ws_weekly.append(row)

    ws_daily = wb.create_sheet(title="اليومية")
    ws_daily.append(headers)
    for row in daily_rows:
        ws_daily.append(row)

    ws_total = wb.create_sheet(title="الكلية")
    ws_total.append(headers)
    for row in total_rows:
        ws_total.append(row)
        
    # =========================
    # Sheet الإحالات (طبقة أولى فقط)
    # =========================
    ws_referrals = wb.create_sheet(title="الإحالات")

    referral_headers = [
        "معرف المستخدم",
        "اسم المستخدم",
        "عدد الإحالات الكلية",
        "عدد الإحالات النشطة",
        "الرصيد"
    ]
    ws_referrals.append(referral_headers)

    with get_connection() as conn:
        cur = conn.cursor()

        # جلب جميع العلاقات (طبقة أولى)
        cur.execute("""
            SELECT referrer_id, invited_id
            FROM referral_relationships
        """)
        relationships = cur.fetchall()

        # بناء خريطة referrer -> [invited_ids]
        referrals_map = {}
        for row in relationships:
            referrer = str(row["referrer_id"])
            invited = str(row["invited_id"])
            referrals_map.setdefault(referrer, []).append(invited)

        # المرور على كل مستخدم
        for user_id, user_data in user_accounts.items():
            user_id = str(user_id)
            username = user_data.get("username", f"user_{user_id}")
            balance = float(user_data.get("balance", 0))

            invited_users = referrals_map.get(user_id, [])
            total_refs = len(invited_users)

            active_refs = 0
            for invited_id in invited_users:
                if is_user_active(invited_id):
                    active_refs += 1

            ws_referrals.append([
                user_id,
                username,
                total_refs,
                active_refs,
                balance
            ])


    # 🔧 تحسين عرض الأعمدة
    for sheet in [ws_weekly, ws_daily, ws_total , ws_referrals]:
        for col in sheet.columns:
            max_len = 0
            col_letter = get_column_letter(col[0].column)
            for cell in col:
                if cell.value:
                    max_len = max(max_len, len(str(cell.value)))
            sheet.column_dimensions[col_letter].width = min(max_len + 2, 50)

    # --- حفظ الملف وإرساله ---
    with tempfile.NamedTemporaryFile(delete=False, suffix=".xlsx") as tmp:
        file_path = tmp.name
        wb.save(file_path)

    try:
        with open(file_path, "rb") as f:
            await query.message.reply_document(
                document=f,
                filename=f"الإحصائيات_{today.strftime('%Y-%m-%d')}.xlsx"
            )
    finally:
        os.unlink(file_path)
