# handlers/admin/promo_management.py

from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
from database import (
    get_promo_code,
    get_all_active_promo_codes,
    get_all_promo_codes,  # ← ستحتاج إضافتها (انظر أدناه)
    add_promo_code,
    delete_promo_code,
    update_promo_code,
)
from constants import (
    AWAITING_PROMO_CODE_NAME,
    AWAITING_PROMO_CODE_VALUE,
    AWAITING_PROMO_CODE_USES,
    AWAITING_NEW_PROMO_VALUE,
    AWAITING_CONFIRM_PUBLISH
)
from config import CHANNEL_ID


# --------------------------------------------------
# دالة مساعدة: جلب جميع الأكواد (حتى غير النشطة)
# --------------------------------------------------
def get_all_promo_codes():
    """جلب جميع الأكواد (حتى تلك ذات uses_left = 0)"""
    try:
        import sqlite3
        from database import get_connection
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT code, value, max_uses, uses_left FROM promo_codes")
            rows = cur.fetchall()
            return {row["code"]: dict(row) for row in rows}
    except sqlite3.Error as e:
        print(f"[ERROR] Failed to fetch all promo codes: {e}")
        return {}


# ===================== إضافة كود هدية (غير منشور) =====================
async def start_add_promo_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    
    await query.edit_message_text(
        "🎁 *إنشاء كود هدية جديد*\n\n"
        "ملاحظة:\n"
        "• الكود سيكون **فعالاً فوراً** ويمكن للأدمن استخدامه\n"
        "• الكود **غير منشور** للمستخدمين حتى يتم نشره في القناة\n\n"
        "أدخل اسم كود الهدية (مثال: WELCOME10):",
        parse_mode='Markdown'
    )
    return AWAITING_PROMO_CODE_NAME

async def receive_promo_code_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
    code_name = update.message.text.strip().upper()
    if not code_name or len(code_name) < 3:
        await update.message.reply_text("❌ الاسم غير صالح. أعد المحاولة:")
        return AWAITING_PROMO_CODE_NAME

    all_codes = get_all_promo_codes()
    if code_name in all_codes:
        await update.message.reply_text(f"❌ الكود '{code_name}' موجود مسبقًا! أدخل اسمًا آخر:")
        return AWAITING_PROMO_CODE_NAME

    context.user_data['promo_code_name'] = code_name
    await update.message.reply_text(f"💰 أدخل قيمة الكود بالليرة السورية (مثال: 1000):")
    return AWAITING_PROMO_CODE_VALUE


async def receive_promo_code_value(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        value = float(update.message.text)
        if value <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ الرجاء إدخال رقم موجب صالح للقيمة:")
        return AWAITING_PROMO_CODE_VALUE

    context.user_data['promo_code_value'] = value
    await update.message.reply_text(
        f"🔢 أدخل الحد الأقصى لعدد مرات استخدام الكود (لكل المستخدمين)."
    )
    return AWAITING_PROMO_CODE_USES


async def receive_promo_code_uses(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        max_uses = int(update.message.text)
        if max_uses <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ الرجاء إدخال عدد صحيح موجب لعدد الاستخدامات:")
        return AWAITING_PROMO_CODE_USES

    code_name = context.user_data['promo_code_name']
    value = context.user_data['promo_code_value']

    # إنشاء الكود بـ uses_left = 0 (غير منشور)
    add_promo_code(
        code=code_name, 
        value=value, 
        max_uses=max_uses, 
        uses_left=max_uses,  # 🟢 **فعال فوراً**
        published=False  # 🟢 **غير منشور للمستخدمين**
    )

    context.user_data.clear()
    await update.message.reply_text(
        f"✅ تم إنشاء الكود بنجاح!\n"
        f"الكود: `{code_name}`\nالقيمة: {int(value):,} ل.س\n"
        f"الاستخدامات: {max_uses} مرة\n"
        "❗ الكود **غير منشور** بعد. استخدم 'نشر الأكواد' لتفعيله.",
        parse_mode='Markdown'
    )
    return ConversationHandler.END


# ===================== نشر الأكواد غير المنشورة =====================
async def start_publish_promo_codes(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    
    all_codes = get_all_promo_codes()
    unpublished = {k: v for k, v in all_codes.items() if v["uses_left"] == 1}
    
    if not unpublished:
        await query.edit_message_text("📭 لا توجد أكواد غير منشورة للنشر.")
        return ConversationHandler.END

    context.user_data['unpublished_codes'] = unpublished
    await query.edit_message_text(
        f"📢 يوجد {len(unpublished)} كود غير منشور.\n"
        "كم عدد الأكواد التي تريد نشرها؟"
    )
    return "AWAITING_PUBLISH_COUNT"


async def receive_publish_count(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        count = int(update.message.text)
        if count <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ أدخل عددًا صحيحًا صالحًا:")
        return "AWAITING_PUBLISH_COUNT"

    unpublished = context.user_data.get('unpublished_codes', {})
    codes_to_publish = list(unpublished.items())[:count]
    
    if not codes_to_publish:
        await update.message.reply_text("❌ لا توجد أكواد كافية للنشر.")
        return ConversationHandler.END

    # رسالة المعاينة
    preview = "🎉 أكواد الهدايا انطلقت!\n"
    preview += "الكود صالح للاستخدام مرة واحدة فقط، كن سريعا! 💵\n\n"
    for code, data in codes_to_publish:
        preview += f"💎 <code>{code}</code> ({int(data['value']):,} ل.س)\n"

    channel_msg = "🎉 أكواد الهدايا انطلقت!\n"
    channel_msg += "الكود صالح للاستخدام مرة واحدة فقط، كن سريعا! 💵\n\n"
    for code, _ in codes_to_publish:
        channel_msg += f"💎 <code>{code}</code>\n"
    channel_msg += "\n✅ رابط البوت: @luffy_ichancy_bot "

    keyboard = [
        [InlineKeyboardButton("✅ نشر", callback_data="confirm_publish_promo")],
        [InlineKeyboardButton("❌ إلغاء", callback_data="promo_codes_menu")]
    ]
    await update.message.reply_text(
        preview, 
        reply_markup=InlineKeyboardMarkup(keyboard), 
        parse_mode='HTML'
    )
    context.user_data['codes_to_publish'] = codes_to_publish
    context.user_data['channel_message'] = channel_msg
    return AWAITING_CONFIRM_PUBLISH


async def confirm_publish_promo(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    codes = context.user_data.get('codes_to_publish', [])
    msg = context.user_data.get('channel_message')

    if not codes or not msg:
        await query.edit_message_text("❌ خطأ: لا توجد أكواد للنشر.")
        return ConversationHandler.END

    # 1. نشر في القناة
    try:
        await context.bot.send_message(chat_id=CHANNEL_ID, text=msg, parse_mode='HTML')
    except Exception as e:
        await query.edit_message_text(f"❌ فشل النشر في القناة: {e}")
        return ConversationHandler.END

    # 2. تحديث كل كود ليصبح uses_left = max_uses (منشور!)
    from database import set_promo_code_uses_left
    for code, data in codes:
        set_promo_code_uses_left(code, data['max_uses'])

    await query.edit_message_text(f"✅ تم نشر {len(codes)} كود في القناة بنجاح.")
    context.user_data.clear()
    return ConversationHandler.END


# ===================== تعديل أكواد (من database.py مباشرة) =====================
async def edit_promo_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    codes = get_all_promo_codes()
    if not codes:
        await query.edit_message_text("📭 لا توجد أكواد.")
        return ConversationHandler.END

    keyboard = [
        [InlineKeyboardButton(f"{code} ({int(data['value']):,} ل.س)", callback_data=f"select_edit_code:{code}")]
        for code, data in codes.items()
    ]
    keyboard.append([InlineKeyboardButton("🔙", callback_data="promo_codes_menu")])
    await query.edit_message_text("✏️ اختر الكود للتعديل:", reply_markup=InlineKeyboardMarkup(keyboard))


async def select_promo_code_to_edit(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    code = query.data.split(':', 1)[1]
    context.user_data['editing_code'] = code
    await query.edit_message_text(f"💰 أدخل القيمة الجديدة للكود `{code}`:")
    return AWAITING_NEW_PROMO_VALUE


async def receive_new_promo_value(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        new_value = float(update.message.text)
        if new_value <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ أدخل رقمًا موجبًا:")
        return AWAITING_NEW_PROMO_VALUE

    code = context.user_data['editing_code']
    # تحديث القيمة فقط (max_uses و uses_left يبقيان كما هما)
    from database import update_promo_code_value
    update_promo_code_value(code, new_value)

    await update.message.reply_text(
        f"✅ تم تحديث الكود `{code}` إلى {int(new_value):,} ل.س",
        parse_mode='Markdown'
    )
    context.user_data.clear()
    return ConversationHandler.END


# ===================== حذف كود =====================
async def start_delete_promo_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    codes = get_all_promo_codes()
    if not codes:
        await query.edit_message_text("📭 لا توجد أكواد للحذف.")
        return
    keyboard = [
        [InlineKeyboardButton(code, callback_data=f"delete_code:{code}")]
        for code in codes.keys()
    ]
    keyboard.append([InlineKeyboardButton("🔙", callback_data="promo_codes_menu")])
    await query.edit_message_text("🗑️ اختر كود للحذف:", reply_markup=InlineKeyboardMarkup(keyboard))


async def confirm_delete_promo_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    code = query.data.split(':', 1)[1]
    delete_promo_code(code)
    await query.edit_message_text(f"✅ تم حذف الكود `{code}`.", parse_mode='Markdown')


# ===================== عرض جميع الأكواد =====================
async def list_promo_codes(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    codes = get_all_promo_codes()
    if not codes:
        text = "📭 لا توجد أكواد."
    else:
        lines = []
        for code, data in codes.items():
            status = "✅ منشور" if data["uses_left"] > 0 else "⏳ غير منشور"
            lines.append(f"`{code}` → {int(data['value']):,} ل.س ({status})")
        text = "📋 جميع أكواد الهدايا:\n\n" + "\n".join(lines)

    keyboard = [[InlineKeyboardButton("🔙", callback_data="promo_codes_menu")]]
    await query.edit_message_text(text, reply_markup=InlineKeyboardMarkup(keyboard), parse_mode='Markdown')


# ===================== إلغاء =====================
async def cancel_promo_operation(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text("❌ تم الإلغاء.")
    return ConversationHandler.END