# handlers/admin/withdraw_approval.py

import datetime
import asyncio
import os
import json
import time
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler

from database import (
    get_connection, get_user_data, update_user_field, add_transaction,
    deduct_referral_on_loss, get_pending_withdrawals, update_pending_withdrawal_status,
    delete_pending_withdrawal , _get_setting
)
from handlers.withdraw.chamcash_withdraw import execute_chamcash_withdrawal 
from handlers.withdraw.cash_syriatel import enqueue_withdraw_request

TRANSACTION_CHANNEL_ID = -1003549611173  # ضع هنا معرف القناة الخاصة بالعمليات
CONFIG_FILE = "data/config.json"  # تأكد من أن المسار صحيح
AWAITING_HAWALA_PROOF = 1  # حالة انتظار صورة إشعار الحوالة

def get_config():
    """قراءة ملف الإعدادات."""
    if not os.path.exists(CONFIG_FILE):
        return {}
    try:
        with open(CONFIG_FILE, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception as e:
        print(f"⚠️ خطأ أثناء قراءة config.json: {e}")
        return {}


async def handle_withdraw_decision(update, context):
    query = update.callback_query
    await query.answer()

    # --- 1. التحقق من الطلب وبياناته ---
    try:
        if ':' not in query.data:
            raise ValueError
        action, request_id = query.data.split(':', 1)
    except Exception:
        await query.edit_message_text("❌ خطأ في تنسيق بيانات الطلب.")
        return

    # حفظ الرسالة الأصلية قبل أي تعديلات
    original_message = query.message.text if query.message else ""

    # جلب الطلب من جدول pending_withdrawals
    pending_request = None
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("SELECT * FROM pending_withdrawals WHERE request_id = ?", (request_id,))
            row = cur.fetchone()
            if row:
                pending_request = dict(row)
    except Exception as e:
        print(f"❌ خطأ في جلب طلب السحب: {e}")
        await query.edit_message_text("❌ خطأ في قاعدة البيانات.")
        return

    if not pending_request or pending_request.get("status") != "pending":
        await query.edit_message_text(original_message + "\n\n⚠️ تم التعامل مع هذا الطلب مسبقاً أو غير موجود.")
        return

    user_id_str = str(pending_request.get("user_id", ""))
    method = pending_request.get("method", "CHAMCASH").upper()
    client_number = pending_request.get("client_number", "")
    
    # 🔥 **الحقول المهمة لـ USDT**
    original_currency = pending_request.get("original_currency", "SYP").upper()
    network = pending_request.get("network", "")
    wallet_address = pending_request.get("wallet_address", "")
    amount_in_usdt = float(pending_request.get("amount_in_usdt", 0))
    
    # 🔥 **التصحيح: تحديد المبلغ الفعلي بناءً على العملة**
    exchange_rate = _get_setting("exchange_rate", 4500)
    
    # جلب بيانات المستخدم
    user_data = get_user_data(user_id_str)
    if not user_data:
        await query.edit_message_text("❌ المستخدم غير موجود.")
        try:
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("DELETE FROM pending_withdrawals WHERE request_id = ?", (request_id,))
                conn.commit()
        except Exception as e:
            print(f"❌ خطأ في حذف طلب مستخدم غير موجود: {e}")
        return

    current_balance = user_data.get("balance", 0.0)

    # --- 2. معالجة الموافقة ---
    if action == "approve_withdraw":
        # 🔥 **تحليل البيانات حسب نوع العملة**
        if original_currency == "USD":
            # حالة الدولار
            amount_usd = float(pending_request.get("requested_amount", 0))
            amount_syp = amount_usd 
            display_amount_original = amount_usd
            currency_display = "دولار"
            
            amount_display_admin = f"{amount_usd:,.2f} دولار ({amount_syp:,.0f} ل.س)"
            amount_display_user = f"{amount_usd:,.2f} دولار"
            
            # حساب الرسوم لـ USD
            fee_percent = get_config().get("WITHDRAWAL_FEE_PERCENT", {}).get("CHAMCASH_USD", 5)
            net_amount_usd = round(amount_usd * (100 - fee_percent) / 100, 2)
            net_amount_syp = net_amount_usd * exchange_rate
            
        elif original_currency == "USDT" or "USDT" in method.upper():
            # حالة USDT - 🔥 **التصحيح: استخدام amount_in_usdt**
            if amount_in_usdt > 0:
                usdt_amount = amount_in_usdt
            else:
                usdt_amount = float(pending_request.get("requested_amount", 0))
                
            amount_syp = usdt_amount * exchange_rate
            display_amount_original = usdt_amount
            currency_display = "USDT"
            
            amount_display_admin = f"{usdt_amount:,.2f} USDT ({amount_syp:,.0f} ل.س)"
            amount_display_user = f"{usdt_amount:,.2f} USDT"
            
            # حساب الرسوم لـ USDT
            fee_percent = get_config().get("WITHDRAWAL_FEE_PERCENT", {}).get("USDT", 10)
            net_amount_usdt = round(usdt_amount * (100 - fee_percent) / 100, 2)
            net_amount_syp = net_amount_usdt * exchange_rate
            
            # 🔥 **تحديد display_client للمحفظة**
            display_client = wallet_address if wallet_address else client_number
            network_display = f" ({network})" if network else ""
            
        elif method == "SYRIATEL":
            # حالة سيرياتيل كاش - المبلغ بالليرة
            amount_syp = float(pending_request.get("requested_amount", 0))
            display_amount_original = amount_syp
            currency_display = "ليرة"
            
            amount_display_admin = f"{amount_syp:,.0f} ليرة"
            amount_display_user = f"{amount_syp:,.0f} ليرة"
            
            # حساب الرسوم لسيرياتيل
            fee_percent = get_config().get("WITHDRAWAL_FEE_PERCENT", {}).get("SYRIATEL", 5)
            net_amount_syp = round(amount_syp * (100 - fee_percent) / 100)
            display_client = client_number
            
        else:
            # حالة الليرة السورية للطرق الأخرى
            amount_syp = float(pending_request.get("requested_amount", 0))
            display_amount_original = amount_syp
            currency_display = "ليرة"
            
            amount_display_admin = f"{amount_syp:,.0f} ليرة"
            amount_display_user = f"{amount_syp:,.0f} ليرة"
            
            # حساب الرسوم للطرق الأخرى
            fee_percent = get_config().get("WITHDRAWAL_FEE_PERCENT", {}).get(method, 5)
            net_amount_syp = round(amount_syp * (100 - fee_percent) / 100)
            display_client = client_number

        # 🔥 **التأكد من أن الرصيد كافي**
        

        try:
            # خصم المبلغ بالليرة السورية من رصيد المستخدم
            
            
            
            # تسجيل المعاملة في transaction_logs
            transaction_data = {
                "user_id": user_id_str,
                "type": "withdraw",
                "amount": amount_syp,  # 🔥 تخزين المبلغ بالليرة
                "method": method.lower(),
                "status": "success",
                "timestamp": time.time(),
                "client_number": client_number,
                "extra": json.dumps({
                    "request_id": request_id,
                    "original_currency": original_currency,
                    "original_amount": display_amount_original,
                    "amount_in_usdt": amount_in_usdt if original_currency == "USDT" else 0,
                    "network": network,
                    "wallet_address": wallet_address,
                    "exchange_rate": exchange_rate,
                    "amount_in_syp": amount_syp,
                    "net_amount_syp": net_amount_syp if 'net_amount_syp' in locals() else 0,
                    "fee_percent": fee_percent if 'fee_percent' in locals() else 0
                })
            }
            
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    INSERT INTO transaction_logs 
                    (user_id, type, amount, method, status, timestamp, client_number, extra) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                """, (
                    transaction_data["user_id"],
                    transaction_data["type"],
                    transaction_data["amount"],
                    transaction_data["method"],
                    transaction_data["status"],
                    transaction_data["timestamp"],
                    transaction_data["client_number"],
                    transaction_data["extra"]
                ))
                conn.commit()
            
            # تحديث وحذف الطلب من جدول pending_withdrawals
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    DELETE FROM pending_withdrawals 
                    WHERE request_id = ?
                """, (request_id,))
                conn.commit()

        except Exception as e:
            print(f"❌ خطأ في معالجة الموافقة: {e}")
            import traceback
            traceback.print_exc()
            await query.edit_message_text(original_message + "\n\n❌ خطأ في معالجة الموافقة.")
            return

        # 🔥 **تحديث حالة has_withdrawn وخسارة أرباح الإحالة**
        


        # 🔥 **معالجة التنفيذ الآلي لسيرياتيل كاش**
        if method == "SYRIATEL":
            if not client_number:
                # إعادة الرصيد إذا البيانات ناقصة
                update_user_field(user_id_str, "balance", current_balance)
                await query.edit_message_text(
                    f"❌ بيانات رمز العميل غير متوفرة للطلب {request_id}. تم إعادة المبلغ."
                )
                return

            # حذف الطلب من pending_withdrawals
            try:
                with get_connection() as conn:
                    cur = conn.cursor()
                    cur.execute(
                        "DELETE FROM pending_withdrawals WHERE request_id = ?", 
                        (request_id,)
                    )
                    conn.commit()
            except Exception as e:
                print(f"❌ خطأ في حذف طلب سيرياتيل: {e}")

            # تسجيل المعاملة بشكل صحيح باستخدام add_transaction
            add_transaction(
                user_id_str,          # معرف المستخدم
                "withdraw_confirmed",           # نوع المعاملة
                amount_syp,           # المبلغ بالليرة
                method.lower(),       # طريقة السحب
                "success",            # حالة المعاملة
                client_number         # رمز العميل
            )


            # إشعار الأدمن
            await query.edit_message_text(
                original_message +
                f"\n\n✅ **تمت الموافقة على طلب سحب سيرياتيل كاش:**\n"
                f"💰 المبلغ: {amount_display_admin}\n"
                f"👤 المستخدم: <code>{user_id_str}</code>\n"
                f"💳 رمز العميل: <code>{client_number}</code>\n"
                f"📊 العملة: {original_currency}\n\n"
                f"⚠️ **ملاحظة:** سيتم تحويل المبلغ يدوياً خلال دقائق.\n"
                f"✅ تم حذف الطلب من قائمة الانتظار."
            )

            # إشعار المستخدم
            user_notification_text = (
                f"✅ تمت الموافقة على طلب سحب سيرياتيل كاش الخاص بك!\n\n"
                f"📋 تفاصيل العملية:\n"
                f"• المبلغ: {amount_display_user}\n"
                f"• رمز العميل: {client_number}\n"
                f"• العملة: {original_currency}\n"
                f"• معرّف الطلب: <code>{request_id}</code>\n\n"
                f"⏳ **سوف يتم تحويل المبلغ خلال دقائق.**\n"
                f"💰 تم خصم {amount_syp:,} ل.س من رصيدك.\n\n"
                f"✅ حالة الطلب: **تمت الموافقة والتحويل قيد التنفيذ**"
            )
            await context.bot.send_message(
                chat_id=int(user_id_str), 
                text=user_notification_text, 
                parse_mode='HTML'
            )
            
            return


            
        elif method in ["CHAMCASH", "CHAMCASH_USD"]:
            if not client_number:
                update_user_field(user_id_str, "balance", current_balance)
                await query.edit_message_text(f"❌ بيانات رمز العميل غير متوفرة للطلب {request_id}. تم إعادة المبلغ.")
                return

            # 🔥 **التعديل: فقط نرسل إشعار ولا ننفذ أي عملية آلية**
            await query.edit_message_text(
                original_message +
                f"\n\n✅ **تمت الموافقة على طلب سحب شام كاش:**\n"
                f"💰 المبلغ: {amount_display_admin}\n"
                f"👤 المستخدم: <code>{user_id_str}</code>\n"
                f"💳 رمز العميل: <code>{client_number}</code>\n"
                f"📊 العملة: {original_currency}\n\n"
                f"⚠️ **ملاحظة:** سيتم تحويل المبلغ يدوياً خلال دقائق.\n"
                f"✅ تم حذف الطلب من قائمة الانتظار."
            )
            
            # إرسال إشعار للمستخدم
            user_notification_text = (
                f"✅ تمت الموافقة على طلب سحب شام كاش الخاص بك!\n\n"
                f"📋 تفاصيل العملية:\n"
                f"• المبلغ: {amount_display_user}\n"
                f"• رمز العميل: {client_number}\n"
                f"• العملة: {original_currency}\n"
                f"• معرّف الطلب: <code>{request_id}</code>\n\n"
                f"⏳ **سوف يتم تحويل المبلغ خلال دقائق.**\n"
                f"💰 تم خصم {amount_syp:,} ل.س من رصيدك.\n\n"
                f"✅ حالة الطلب: **تمت الموافقة والتحويل قيد التنفيذ**"
            )
            
            await context.bot.send_message(
                chat_id=int(user_id_str), 
                text=user_notification_text, 
                parse_mode='HTML'
            )
            
            return

        elif original_currency == "USDT" or "USDT" in method.upper():
            # 🔥 **إشعار USDT للمستخدم**
            user_message = (
                f"✅ تمت الموافقة على طلب سحب USDT الخاص بك!\n\n"
                f"📋 تفاصيل العملية:\n"
                f"• المبلغ: {display_amount_original:,.2f} USDT\n"
                f"• بعد الخصم {fee_percent}%: {net_amount_usdt:,.2f} USDT\n"
                f"• الشبكة: {network}{network_display}\n"
                f"• ما يعادل: {amount_syp:,.0f} ل.س (سعر الصرف: {exchange_rate})\n"
                f"• المحفظة: <code>{display_client}</code>\n"
                f"• معرّف الطلب: <code>{request_id}</code>\n\n"
                f"💰 تم خصم {amount_syp:,.0f} ل.س من رصيدك.\n"
                f"📊 رصيدك الحالي: {new_balance:,.0f} ل.س\n\n"
                f"✅ حالة الطلب: **تمت الموافقة**"
            )
            
            # 🔥 **إشعار USDT للإدمن**
            admin_message = (
                f"{original_message}\n\n"
                f"✅ **تمت الموافقة على طلب سحب USDT:**\n"
                f"👤 المستخدم: <code>{user_id_str}</code>\n"
                f"💵 المبلغ: {amount_display_admin}\n"
                f"📉 بعد خصم {fee_percent}%: {net_amount_usdt:,.2f} USDT ({net_amount_syp:,.0f} ل.س)\n"
                f"🌐 الشبكة: {network}\n"
                f"💳 المحفظة: <code>{display_client}</code>\n"
                f"📊 الرصيد السابق: {current_balance:,.0f} ل.س\n"
                f"📊 الرصيد الجديد: {new_balance:,.0f} ل.س\n"
                f"✅ تم خصم {amount_syp:,.0f} ل.س من الرصيد."
            )
            
            await query.edit_message_text(admin_message, parse_mode="HTML")
            await context.bot.send_message(
                chat_id=int(user_id_str), 
                text=user_message, 
                parse_mode='HTML'
            )
            
        else:
            # طرق السحب اليدوية الأخرى
            await query.edit_message_text(
                original_message + f"\n\n✅ تمت الموافقة على طلب سحب {amount_display_admin} وخصم المبلغ (تنفيذ يدوي/غير آلي)."
            )
            
            await context.bot.send_message(
                chat_id=int(user_id_str),
                text=f"✅ تم خصم {amount_display_user} من محفظتك بعد موافقة الإدارة."
            )
            
        return

    # --- 3. معالجة الرفض ---
    elif action == "reject_withdraw":
        try:
            # 🔥 **تحليل البيانات حسب نوع العملة للرفض**
            if original_currency == "USD":
                amount_syp = float(pending_request.get("requested_amount", 0))
                amount_usd = amount_syp / exchange_rate

                amount_display_user = f"{amount_usd:,.2f} دولار"
                amount_display_admin = f"{amount_usd:,.2f} دولار ({amount_syp:,.0f} ل.س)"

                
            elif original_currency == "USDT" or "USDT" in method.upper():
                # جلب المبلغ الأصلي بالـ USDT من قاعدة البيانات
                usdt_amount = float(pending_request.get("requested_amount", 0))
                
                # المبلغ بالليرة السورية
                amount_syp = usdt_amount * exchange_rate

                # عرض المبلغ للمستخدم
                amount_display_user = f"{usdt_amount:,.2f} USDT"
                amount_display_admin = f"{usdt_amount:,.2f} USDT ({amount_syp:,.0f} ل.س)"

            else:
                # المبالغ الأخرى (مثل SYP)
                amount_syp = float(pending_request.get("requested_amount", 0))
                amount_display_user = f"{amount_syp:,.0f} ل.س"
                amount_display_admin = f"{amount_syp:,.0f} ل.س"

            
            # إعادة المبلغ إلى رصيد المستخدم
            new_balance = current_balance + amount_syp
            update_user_field(user_id_str, "balance", new_balance)
            
            
            
            # تسجيل معاملة الإرجاع
            add_transaction(
                user_id=user_id_str,
                tx_type="refund",
                amount=amount_syp,
                method=f"{method.lower()}_rejected",
                status="success",
                client_number=client_number
            )
            
            # إضافة تفاصيل الإرجاع
            try:
                with get_connection() as conn:
                    cur = conn.cursor()
                    cur.execute("""
                        UPDATE transaction_logs 
                        SET extra = ? 
                        WHERE id = (SELECT MAX(id) FROM transaction_logs WHERE user_id = ? AND type = 'refund')
                    """, (json.dumps({
                        "request_id": request_id,
                        "original_currency": original_currency,
                        "original_amount": usdt_amount if original_currency == "USDT" else amount_usd if original_currency == "USD" else amount_syp,
                        "amount_in_usdt": amount_in_usdt,
                        "network": network,
                        "wallet_address": wallet_address,
                        "refunded_amount_syp": amount_syp,
                        "exchange_rate": exchange_rate,
                        "refund_reason": "rejected_by_admin",
                        "previous_balance": current_balance,
                        "new_balance": new_balance
                    }), user_id_str))
                    conn.commit()
            except Exception as e:
                print(f"⚠️ خطأ في تحديث بيانات الإرجاع: {e}")
            
            # حذف الطلب من جدول pending_withdrawals
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    DELETE FROM pending_withdrawals 
                    WHERE request_id = ?
                """, (request_id,))
                deleted_count = cur.rowcount
                conn.commit()
                
            # تحديث أي معاملة سحب مرتبطة بهذا الطلب
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE transaction_logs 
                    SET status = 'cancelled',
                        extra = json_set(COALESCE(extra, '{}'), '$.rejected', true, '$.refunded', true)
                    WHERE user_id = ? 
                    AND extra LIKE ?
                    AND status = 'pending'
                """, (user_id_str, f'%"request_id":"{request_id}"%'))
                updated_count = cur.rowcount
                conn.commit()
                
            # إرسال الرسائل
            await query.edit_message_text(
                original_message + f"\n\n❌ تم رفض طلب سحب {amount_display_admin}.\n"
                f"💰 تم إعادة {amount_syp:,.0f} ل.س إلى رصيد المستخدم.\n"
                f"📊 الرصيد الجديد: {new_balance:,.0f} ل.س"
            )
            
            # 🔥 **رسالة الرفض للمستخدم حسب العملة**
            if original_currency == "USDT" or "USDT" in method.upper():
                user_reject_message = (
                    f"❌ تم رفض طلب سحب USDT الخاص بك.\n\n"
                    f"💵 المبلغ: {amount_display_user}\n"
                    f"💰 ما يعادل: {amount_syp:,.0f} ل.س\n"
                    f"🌐 الشبكة: {network if network else 'غير محدد'}\n\n"
                    f"✅ **تم إعادة {amount_syp:,.0f} ل.س إلى رصيدك.**\n"
                    f"📊 رصيدك الحالي: {new_balance:,.0f} ل.س\n\n"
                    f"🆔 رقم الطلب: {request_id}"
                )
            else:
                user_reject_message = (
                    f"❌ تم رفض طلب سحبك بقيمة {amount_display_user} من قبل الإدارة.\n\n"
                    f"💰 **تم إعادة المبلغ ({amount_syp:,.0f} ل.س) إلى رصيدك.**\n"
                    f"📊 رصيدك الحالي: {new_balance:,.0f} ل.س\n\n"
                    f"🆔 رقم الطلب: {request_id}"
                )
            
            await context.bot.send_message(
                chat_id=int(user_id_str),
                text=user_reject_message,
                parse_mode="HTML"
            )
            
            print(f"✅ تم معالجة رفض السحب بنجاح للطلب {request_id}")
                
        except Exception as e:
            print(f"❌ خطأ في معالجة الرفض: {e}")
            import traceback
            traceback.print_exc()
            await query.edit_message_text(original_message + f"\n\n❌ خطأ في معالجة الرفض: {str(e)}")

async def handle_hawala_withdraw_decision(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    دالة لمعالجة طلبات الحوالة (موافقة أو رفض) من قبل الأدمن.
    """
    query = update.callback_query
    await query.answer()

    # التحقق من صحة بيانات الزر
    try:
        if ':' not in query.data:
            raise ValueError
        action, request_id = query.data.split(':', 1)
    except Exception:
        await query.edit_message_text("❌ خطأ في تنسيق بيانات الطلب.")
        return ConversationHandler.END

    # جلب طلب الحوالة من قاعدة البيانات
    pending_request = None
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT * FROM pending_withdrawals 
                WHERE request_id = ? AND status = 'pending'
            """, (request_id,))
            row = cur.fetchone()
            if row:
                pending_request = dict(row)
    except Exception as e:
        print(f"❌ خطأ في جلب طلب الحوالة: {e}")

    if not pending_request:
        await query.edit_message_text(query.message.text + "\n\n⚠️ تم التعامل مع هذا الطلب مسبقاً أو غير موجود.")
        return ConversationHandler.END

    user_id = pending_request.get('user_id', '')
    original_message = query.message.text

    user_account = get_user_data(str(user_id))
    if not user_account:
        await query.edit_message_text("❌ المستخدم غير موجود.")
        try:
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("DELETE FROM pending_withdrawals WHERE request_id = ?", (request_id,))
                conn.commit()
        except Exception as e:
            print(f"❌ خطأ في حذف طلب مستخدم غير موجود: {e}")
        return ConversationHandler.END

    if action == "hawala_approve":
        # تخزين بيانات الطلب لانتظار صورة الإشعار
        context.user_data["hawala_proof_request"] = {
            "user_id": user_id,
            "request_id": request_id,
            "amount": pending_request.get('requested_amount', 0)
        }
        
        await query.edit_message_text(
            original_message + "\n\n✅ تمت الموافقة مؤقتًا. الرجاء الآن إرسال صورة إشعار التحويل لإرسالها للعميل."
        )
        
        return AWAITING_HAWALA_PROOF  # ← سيتم التعامل مع خصم الرصيد وإرسال الصورة في receive_hawala_proof

    elif action == "hawala_reject":
        try:
            # تحديث حالة الطلب إلى مرفوض
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE pending_withdrawals 
                    SET status = 'rejected' 
                    WHERE request_id = ?
                """, (request_id,))
                conn.commit()

            # إشعار المستخدم
            try:
                await context.bot.send_message(
                    chat_id=int(user_id),
                    text=f"❌ تم رفض طلب الحوالة بقيمة {pending_request.get('requested_amount', 0):,} ل.س من قبل الإدارة."
                )
            except Exception as e:
                print(f"⚠️ فشل إرسال إشعار رفض للمستخدم {user_id}: {e}")

            # إشعار القناة
            try:
                channel_text = (
                    f"🔴 سحب حوالة مرفوض:\n"
                    f"👤 المستخدم: {user_id}\n"
                    f"💰 المبلغ: {pending_request.get('requested_amount', 0):,} ل.س\n"
                    f"📌 الحالة: مرفوض"
                )
                await context.bot.send_message(TRANSACTION_CHANNEL_ID, channel_text)
            except Exception as e:
                print(f"⚠️ فشل إرسال عملية الحوالة المرفوضة للقناة: {e}")

            await query.edit_message_text(original_message + "\n\n❌ تم رفض طلب الحوالة.")
            
        except Exception as e:
            print(f"❌ خطأ في معالجة رفض الحوالة: {e}")
            await query.edit_message_text("❌ خطأ في معالجة الرفض.")
            
        return ConversationHandler.END


async def show_pending_withdrawals(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """عرض جميع طلبات السحب المعلقة مع دعم USDT."""
    query = update.callback_query
    await query.answer()

    # جلب جميع طلبات السحب المعلقة
    pending_requests = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT * FROM pending_withdrawals 
                WHERE status = 'pending' 
                ORDER BY timestamp ASC
            """)
            rows = cur.fetchall()
            pending_requests = [dict(row) for row in rows]
    except Exception as e:
        print(f"❌ خطأ في جلب طلبات السحب المعلقة: {e}")
        await query.edit_message_text("❌ خطأ في جلب البيانات.")
        return

    if not pending_requests:
        await query.edit_message_text("✅ لا توجد أي طلبات سحب معلقة حاليًا.")
        return
    
    config = get_config()
    fee_config = config.get("WITHDRAWAL_FEE_PERCENT", {})
    exchange_rate = _get_setting("exchange_rate", 4500)
    
    text = "💸 <b>طلبات السحب المعلقة:</b>\n\n"
    keyboard = []

    for i, req in enumerate(pending_requests, start=1):
        user_id = req.get('user_id', 'غير معروف')
        method = req.get('method', '').lower()
        raw_timestamp = req.get('timestamp', 0)
        original_currency = req.get('original_currency', 'SYP').upper()

        # معالجة الوقت
        if isinstance(raw_timestamp, (int, float)):
            timestamp = datetime.datetime.fromtimestamp(raw_timestamp).strftime('%Y-%m-d %H:%M:%S')
        else:
            timestamp = str(raw_timestamp)

        # 🔴 **الحصول على البيانات**
        amount_in_syp = float(req.get('requested_amount', 0))  # المبلغ بالليرة
        amount_in_usdt = float(req.get('amount_in_usdt', 0))
        client_number = req.get('client_number', 'غير محدد')

        # ------------------------------
        # 🟢 حساب نسبة الخصم
        # ------------------------------
        method_key = (
            "USDT" if "usdt" in method or "trc20" in method or "bep20" in method or "erc20" in method
            else "CHAMCASH_USD" if "chamcash_usd" in method
            else "CHAMCASH" if "chamcash" in method
            else "SYRIATEL" if "syriatel" in method
            else "HAWALA" if "hawala" in method or "bank" in method
            else method.upper()
        )

        fee_percent = fee_config.get(method_key, 0)
        fee_amount = amount_in_syp * (fee_percent / 100)
        amount_after_fee = amount_in_syp - fee_amount

        fee_text = (
            f"💵 قبل الخصم: {amount_in_syp:,.0f} ل.س\n"
            f"💸 نسبة الخصم: {fee_percent}%\n"
            f"💰 قيمة الخصم: {fee_amount:,.0f} ل.س\n"
            f"🟢 بعد الخصم: {amount_after_fee:,.0f} ل.س\n"
        )

        # ------------------------------
        # 🔴 تحديد نوع السحب وعرض المبلغ
        # ------------------------------

        if "usdt" in method or "trc20" in method or "bep20" in method or "erc20" in method:
            # حالة USDT
            usdt_amount = amount_in_usdt if amount_in_usdt > 0 else amount_in_syp / exchange_rate
            amount_in_syp = usdt_amount * exchange_rate
            amount_display = f"{usdt_amount:.2f} USDT ({amount_in_syp:,.0f} ل.س)"

            if "trc20" in method:
                network = "USDT (TRC20)"
            elif "bep20" in method:
                network = "USDT (BEP20)"
            elif "erc20" in method:
                network = "USDT (ERC20)"
            else:
                network = "USDT"

            client_display = f"💳 المحفظة: <code>{client_number}</code>\n" \
                if client_number.startswith(('T','0x')) else f"💳 العميل: <code>{client_number}</code>\n"

            approve_callback = f"approve_withdraw_usdt:{req['request_id']}"
            reject_callback = f"reject_withdraw_usdt:{req['request_id']}"

        elif "chamcash_usd" in method or (original_currency == "USD" and "chamcash" in method):
            # 🔴 **حالة الدولار - التصحيح هنا**
            # نحتاج لحساب المبلغ الأصلي بالدولار
            try:
                # إذا كان المبلغ بالدولار محفوظاً في amount_in_usdt
                if amount_in_usdt > 0:
                    usd_amount = amount_in_usdt
                else:
                    # حساب عكسي: الليرة ÷ سعر الصرف = الدولار
                    usd_amount = amount_in_syp / exchange_rate
                
                amount_display = f"{usd_amount:.2f} دولار ({amount_in_syp:,.0f} ل.س)"
            except:
                # إذا فشل الحساب، نعرض بالليرة فقط
                amount_display = f"{amount_in_syp:,.0f} ل.س"
                usd_amount = 0

            network = "ChamCash (USD)"
            client_display = f"💳 رقم العميل: <code>{client_number}</code>\n"
            approve_callback = f"approve_withdraw:{req['request_id']}"
            reject_callback = f"reject_withdraw:{req['request_id']}"

        elif "chamcash" in method:
            # حالة شام كاش بالليرة
            amount_display = f"{int(amount_in_syp):,} ل.س"
            network = "ChamCash (SYP)"
            client_display = f"💳 رقم العميل: <code>{client_number}</code>\n"
            approve_callback = f"approve_withdraw:{req['request_id']}"
            reject_callback = f"reject_withdraw:{req['request_id']}"

        elif "syriatel" in method:
            amount_display = f"{int(amount_in_syp):,} ل.س"
            network = "Syriatel"
            client_display = f"💳 رقم العميل: <code>{client_number}</code>\n"
            approve_callback = f"approve_withdraw:{req['request_id']}"
            reject_callback = f"reject_withdraw:{req['request_id']}"

        elif "hawala" in method or "bank" in method:
            amount_display = f"{int(amount_in_syp):,} ل.س"
            network = "حوالة مصرفية"
            client_display = f"💳 معلومات: <code>{client_number}</code>\n"
            approve_callback = f"hawala_approve:{req['request_id']}"
            reject_callback = f"hawala_reject:{req['request_id']}"

        else:
            amount_display = f"{int(amount_in_syp):,} ل.س"
            network = method.upper()
            client_display = f"💳 رقم العميل: <code>{client_number}</code>\n"
            approve_callback = f"approve_withdraw:{req['request_id']}"
            reject_callback = f"reject_withdraw:{req['request_id']}"

        # ------------------------------
        # 🔵 النص النهائي
        # ------------------------------

        text += (
            f"🧾 <b>طلب {i}</b>\n"
            f"👤 المستخدم: <code>{user_id}</code>\n"
            f"💰 المبلغ: {amount_display}\n"
            f"{fee_text}"
            f"🌐 الشبكة: {network}\n"
            f"{client_display}"
            f"🕒 التاريخ: {timestamp}\n"
            f"🔧 الطريقة: <code>{method.upper()}</code>\n\n"
        )

        keyboard.append([
            InlineKeyboardButton(f"✅ قبول {i}", callback_data=approve_callback),
            InlineKeyboardButton(f"❌ رفض {i}", callback_data=reject_callback)
        ])

    keyboard.append([InlineKeyboardButton("🔙 العودة للوحة الأدمن", callback_data="admin_panel")])

    reply_markup = InlineKeyboardMarkup(keyboard)

    await query.edit_message_text(
        text=text,
        reply_markup=reply_markup,
        parse_mode="HTML"
    )

async def receive_hawala_proof(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """استقبال صورة إشعار الحوالة من الأدمن."""
    if 'hawala_proof_request' not in context.user_data:
        await update.message.reply_text("❌ لا يوجد طلب حوالة قيد الانتظار.")
        return ConversationHandler.END

    request_data = context.user_data['hawala_proof_request']
    user_id = request_data['user_id']
    request_id = request_data['request_id']
    amount = request_data['amount']

    if not update.message.photo:
        await update.message.reply_text("❌ الرجاء إرسال صورة إشعار التحويل.")
        return AWAITING_HAWALA_PROOF

    # جلب بيانات المستخدم
    user_account = get_user_data(str(user_id))
    if not user_account:
        await update.message.reply_text("❌ المستخدم غير موجود.")
        context.user_data.pop('hawala_proof_request', None)
        return ConversationHandler.END

    current_balance = user_account.get('balance', 0)
    amount_syr = amount  # الحوالة دائماً بالليرة السورية

    if current_balance < amount_syr:
        await update.message.reply_text("❌ الرصيد غير كافي.")
        context.user_data.pop('hawala_proof_request', None)
        return ConversationHandler.END

    # خصم الرصيد
    try:
        update_user_field(str(user_id), "balance", current_balance - amount_syr)
        
        # تسجيل المعاملة
        add_transaction(
            user_id=str(user_id),
            tx_type="withdraw",
            amount=amount_syr,
            method="hawala",
            status="success"
        )
        
        # تحديث حالة الطلب
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                UPDATE pending_withdrawals 
                SET status = 'approved' 
                WHERE request_id = ?
            """, (request_id,))
            conn.commit()

        # تحديث حالة has_withdrawn
        update_user_field(str(user_id), "has_withdrawn", 1)
        
        # خصم أرباح الإحالة

    except Exception as e:
        print(f"❌ خطأ في معالجة الحوالة: {e}")
        await update.message.reply_text("❌ خطأ في معالجة الحوالة.")
        context.user_data.pop('hawala_proof_request', None)
        return ConversationHandler.END

    # إرسال الصورة للمستخدم
    try:
        photo_file = await update.message.photo[-1].get_file()
        await context.bot.send_photo(
            chat_id=int(user_id),
            photo=photo_file.file_id,
            caption=f"✅ تمت الموافقة على طلب الحوالة.\n\n💰 تم تحويل مبلغ: {amount_syr:,} ل.س\n📸 هذا هو إشعار التحويل:"
        )
    except Exception as e:
        print(f"⚠️ فشل إرسال الصورة للمستخدم: {e}")
        await context.bot.send_message(
            chat_id=int(user_id),
            text=f"✅ تمت الموافقة على طلب الحوالة.\n💰 تم تحويل مبلغ: {amount_syr:,} ل.س"
        )

    # إشعار الأدمن
    await update.message.reply_text("✅ تم إرسال إشعار التحويل للمستخدم وخصم المبلغ.")

    # إشعار القناة
    try:
        channel_text = (
            f"🟢 سحب حوالة معتمد:\n"
            f"👤 المستخدم: {user_id}\n"
            f"💰 المبلغ: {amount_syr:,} ل.س\n"
            f"📌 الحالة: مكتمل"
        )
        await context.bot.send_message(TRANSACTION_CHANNEL_ID, channel_text)
    except Exception as e:
        print(f"⚠️ فشل إرسال عملية الحوالة للقناة: {e}")

    # تنظيف البيانات المؤقتة
    context.user_data.pop('hawala_proof_request', None)
    
    return ConversationHandler.END


async def cancel_hawala_proof(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """إلغاء عملية إرسال إشعار الحوالة."""
    context.user_data.pop('hawala_proof_request', None)
    await update.message.reply_text("❌ تم إلغاء عملية إرسال إشعار الحوالة.")
    return ConversationHandler.END


# ===============================================
# دوال مساعدة تحتاج إلى إضافتها في database.py
# ===============================================

# أضف هذه الدوال إلى database.py إذا لم تكن موجودة:

def get_pending_withdrawals():
    """جلب جميع طلبات السحب المعلقة."""
    requests = []
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT * FROM pending_withdrawals 
                WHERE status = 'pending' 
                ORDER BY timestamp ASC
            """)
            rows = cur.fetchall()
            requests = [dict(row) for row in rows]
    except Exception as e:
        print(f"❌ خطأ في جلب طلبات السحب المعلقة: {e}")
    return requests


def update_pending_withdrawal_status(request_id: str, status: str):
    """تحديث حالة طلب سحب معلق."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                UPDATE pending_withdrawals 
                SET status = ? 
                WHERE request_id = ?
            """, (status, request_id))
            conn.commit()
            return True
    except Exception as e:
        print(f"❌ خطأ في تحديث حالة طلب السحب: {e}")
        return False


def delete_pending_withdrawal(request_id: str):
    """حذف طلب سحب معلق."""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("DELETE FROM pending_withdrawals WHERE request_id = ?", (request_id,))
            conn.commit()
            return True
    except Exception as e:
        print(f"❌ خطأ في حذف طلب السحب: {e}")
        return False