import requests
import json
import time
import ssl
import random
from requests.adapters import HTTPAdapter
from urllib3.util.ssl_ import create_urllib3_context
from urllib3.util.retry import Retry
import urllib3
import asyncio
from config import ACCOUNTS, get_config
from database import (
    get_user_data,
    create_local_user,
    check_used_transaction,
    add_used_transaction,
    add_transaction,
    get_user_balance,
    
)
from handlers.referral.referral_service import process_referral_for_deposit
# ----------------------------- إعداد SSL والتحذيرات
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# ----------------------------- إعدادات محسنة للـ Session
def _create_robust_session():
    """إنشاء جلسة HTTP قوية"""
    session = requests.Session()
    
    # إعداد Retry محسّن - ✅ التصحيح هنا
    retry_strategy = Retry(
        total=3,
        backoff_factor=0.5,
        status_forcelist=[429, 500, 502, 503, 504],  # ✅ status_forcelist (بـ e)
        allowed_methods=["POST"],
        raise_on_status=False
    )
    
    class SyriatelTLSAdapter(HTTPAdapter):
        def init_poolmanager(self, *args, **kwargs):
            context = create_urllib3_context()
            context.set_ciphers("DEFAULT@SECLEVEL=1")
            context.check_hostname = False
            context.verify_mode = ssl.CERT_NONE
            kwargs['ssl_context'] = context
            return super().init_poolmanager(*args, **kwargs)
    
    adapter = SyriatelTLSAdapter(max_retries=retry_strategy)
    session.mount("https://", adapter)
    
    # تحسين الـ Headers
    session.headers.update({
        "User-Agent": "Dalvik/2.1.0 (Linux; U; Android 12; SM-A536E Build/d9f5a19.0)",
        "Accept-Encoding": "gzip",
        "Connection": "Keep-Alive",
        "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
    })
    
    return session
# جلسة عالمية
session = _create_robust_session()

# ----------------------------- دوال مساعدة محسنة
device_id = "00000000-0835-1d20-ffff-ffffef05ac4a"
app_version = "5.6.0"
mobile_manufacturer = "Samsung"
mobile_model = "SM-A536E"
system_version = "Android v12"
lang = "1"
history_url = "https://cash-api.syriatel.sy/Wrapper/app/7/SS2MTLGSM/ePayment/customerHistory"

def _fetch_page_safe(account_id, hash_history, page, retry_count=2):
    """جلب صفحة واحدة بأمان مع إعادة المحاولة"""
    payload = {
        "appVersion": app_version,
        "pageNumber": str(page),
        "searchGsmOrSecret": "",
        "type": "2",
        "systemVersion": system_version.replace(" ", "+"),
        "deviceId": device_id,
        "userId": account_id,
        "sortType": "1",
        "mobileManufaturer": mobile_manufacturer,
        "mobileModel": mobile_model,
        "channelName": "4",
        "lang": lang,
        "hash": hash_history,
        "status": "2"
    }
    
    for attempt in range(retry_count):
        try:
            # إضافة تأخير عشوائي بين المحاولات لتخفيف الحمل
            if attempt > 0:
                delay = random.uniform(0.5, 2.0)
                time.sleep(delay)
            
            # Timeout أقصر ولكن مع فترتين مختلفتين
            timeout = (8, 20) if attempt == 0 else (5, 15)
            
            response = session.post(
                history_url, 
                data=payload, 
                headers=session.headers, 
                verify=False, 
                timeout=timeout
            )
            
            # تحقق من حالة HTTP
            if response.status_code != 200:
                print(f"⚠️ حالة HTTP غير متوقعة {response.status_code} للمحاولة {attempt + 1}")
                continue
            
            # محاولة تحليل JSON
            try:
                data = response.json()
            except:
                # إذا فشل JSON، قد تكون الاستجابة نصية
                if "code" in response.text:
                    try:
                        data = json.loads(response.text)
                    except:
                        print(f"❌ فشل تحليل JSON للمحاولة {attempt + 1}")
                        continue
                else:
                    print(f"❌ استجابة غير متوقعة للمحاولة {attempt + 1}")
                    continue
            
            # تحقق من هيكل البيانات
            if not isinstance(data, dict) or data.get("code") != "1":
                print(f"⚠️ code غير صالح: {data.get('code')}")
                continue
            
            return data.get("data", {}).get("data", [])
            
        except requests.exceptions.Timeout:
            print(f"⏰ انتهت مهلة المحاولة {attempt + 1}")
            continue
        except requests.exceptions.ConnectionError:
            print(f"🔌 خطأ اتصال في المحاولة {attempt + 1}")
            continue
        except Exception as e:
            print(f"❌ خطأ غير متوقع في المحاولة {attempt + 1}: {e}")
            continue
    
    return None

def fetch_all_history(account_id, hash_history, max_pages=3):
    """جلب آخر 5 عمليات فقط من الصفحة الأولى (لكل حساب)"""
    page_data = _fetch_page_safe(account_id, hash_history, page=1, retry_count=2)
    if page_data is None:
        return []
    return page_data[:5] 

def check_transaction(transaction_no: str, amount: int):
    """نسخة محسنة: البحث عن العملية"""
   
    
    # توزيع الطلبات على الحسابات بشكل عشوائي
    accounts = list(ACCOUNTS)
    random.shuffle(accounts)
    
    for acc in accounts:
        account_id = acc["account_id"]
        hash_history = acc["hash"]
        number = acc["number"]

        
        
        try:
            all_history = fetch_all_history(account_id, hash_history)
        except Exception as e:
            print(f"⚠️ خطأ في جلب سجل الحساب {number}: {e}")
            continue
        
        if not all_history:
            print(f"📭 لا توجد سجلات للحساب: {number}")
            continue
        
        
        
        # بحث أسرع مع break عند العثور
        for trx in all_history:
            trx_no = trx.get("transactionNo")
            
            if trx_no == transaction_no:
                
                
                # تحقق سريع من المبلغ
                try:
                    trx_amount = int(trx.get("amount") or trx.get("net") or 0)
                except:
                    trx_amount = 0
                
                # تحقق من الحالة
                trx_status = trx.get("status")
                if trx_status == "0":
                    trx["fraud"] = True
                    print("⚠️ العملية مشبوهة (fraud)")
                    return trx
                
                if trx_amount == int(amount):
                    
                    return trx
                else:
                    
                    return None  # نهاية البحث
        
        print(f"❌ العملية غير موجودة في الحساب: {number}")
    
    print(f"❌ لم يتم العثور على العملية: {transaction_no}")
    return None

# ----------------------------- الدالة الرئيسية محسنة
async def credit_wallet(transaction_no: str, amount: int, user_id: str , bot):
    """دالة محسنة لإضافة رصيد مع تحسين الأداء"""
    
    
    # 1. التحقق من عدم تكرار العملية أولاً (الأسرع)
    if await check_used_transaction(transaction_no):
        
        return False

    # 2. البحث عن العملية
    start_time = time.time()
    trx = check_transaction(transaction_no, amount)
    search_time = time.time() - start_time
    
    
    if not trx:
        print("❌ لم يتم العثور على العملية أو المبلغ غير مطابق.")
        return False

    # 3. تحقق من حالة العملية
    if trx.get("fraud"):
        print("❌ العملية مشبوهة")
        return False

    # 4. التأكد من وجود الحساب
    user_account = get_user_data(user_id)
    if not user_account:
        print(f"🆕 إنشاء حساب جديد للمستخدم: {user_id}")
        username = f"user{user_id}"
        if not create_local_user(user_id, username):
            print("❌ فشل إنشاء حساب للمستخدم")
            return False
        user_account = get_user_data(user_id)
        if not user_account:
            print("❌ فشل جلب بيانات الحساب بعد الإنشاء")
            return False

    # 5. تحقق من الحد الأدنى
    cfg = get_config()
    MINIMUM_DEPOSIT = cfg.get("MINIMUM_DEPOSIT", {}).get("SYRIATEL", 0)
    
    if amount < MINIMUM_DEPOSIT:
        print(f"❌ المبلغ أقل من الحد الأدنى ({MINIMUM_DEPOSIT})")
        return False

    # 6. معالجة الإيداع
    try:
        # إضافة المعاملة
        success = add_transaction(
            user_id=user_id, 
            tx_type="deposit", 
            amount=amount, 
            method="syriatel", 
            status="success", 
            client_number=transaction_no
        )
        
        if not success:
            print("❌ فشل تسجيل المعاملة")
            return False
            
      

        # تسجيل العملية كمستخدمة
        await add_used_transaction(
            reference_id=transaction_no, 
            amount=amount, 
            method="syriatel", 
            user_id=user_id
        )
      

        


        # جلب الرصيد الجديد
        new_balance = get_user_balance(user_id)
        
        return True
        
    except Exception as e:
        print(f"❌ فشل في معالجة العملية: {e}")
        return False

# ----------------------------- نفس الدالة المساعدة للتوافق
def credit_wallet_sync(transaction_no: str, amount: int, user_id: str):
    """نفس الدالة للتوافق مع الكود القديم"""
    return asyncio.run(credit_wallet(transaction_no, amount, user_id))

# ----------------------------- دالة اختبارية بسيطة
async def test_api_response():
    """اختبار بسيط لسرعة استجابة API"""
    if not ACCOUNTS:
        print("❌ لا توجد حسابات")
        return
    
    acc = ACCOUNTS[0]
    start_time = time.time()
    
    try:
        page_data = _fetch_page_safe(acc["account_id"], acc["hash"], 1)
        response_time = time.time() - start_time
        
        if page_data is not None:
            print(f"✅ API متجاوب - وقت الاستجابة: {response_time:.2f} ثانية")
            print(f"📊 عدد العمليات في الصفحة الأولى: {len(page_data)}")
        else:
            print(f"❌ API لا يستجيب - حاول بعد قليل")
            
    except Exception as e:
        print(f"❌ خطأ في اختبار API: {e}")