import time
from decimal import Decimal
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler

from database import get_user_data 
from config import ADMIN_IDS, DEPOSIT_INFO, PAYMENT_METHODS, OPERATIONS_CHANNEL_ID, get_config , API_BASE_URL , ARAB_PAYMENT_API_KEY
from constants import (
    AWAITING_DEPOSIT_METHOD,
    AWAITING_DEPOSIT_AMOUNT,
    AWAITING_DEPOSIT_PROOF
)
from database import _get_setting
from database import (
    _get_setting, get_user_data, update_user_balance, 
    add_transaction_log, add_admin_history, 
    check_used_transaction, add_used_transaction
)
from handlers.deposit.syriatel_checker import credit_wallet , check_transaction
from .chamcash_checker import run_auto_deposit_check
from handlers.deposit.verify_tx_v2_anylog import verify_tx
import httpx
from threading import Thread
import asyncio
from database  import add_transaction , add_admin_history
CANCEL_NOTE = "\n\n⚠️ <b>ملاحظة:</b> يرجى إلغاء العملية قبل الضغط على أي زر آخر من القائمة."

def is_deposit_method_enabled(method_key: str) -> bool:
    return _get_setting(f"{method_key}_enabled", True)


def is_deposit_withdraw_enabled():
    # استيراد الدالة من database.py
    from database import _get_setting
    
    # الحصول على القيمة من SQLite
    enabled = _get_setting("deposit_withdraw_enabled", True)
    
    # التأكد من أنها boolean
    if isinstance(enabled, str):
        return enabled.lower() == 'true'
    return bool(enabled)

async def get_arab_payment_currencies():
    
    """الحصول على العملات المتاحة من Vlahalla Bot."""
    # المسار: /availableCurrencies
    url = f"{API_BASE_URL}/availableCurrencies"
    headers = {"Authorization": f"Bearer {ARAB_PAYMENT_API_KEY}"}
    try:
        # استخدام httpx.AsyncClient ليكون الطلب غير متوقف (Async)
        async with httpx.AsyncClient(timeout=10) as client:
            response = await client.get(url, headers=headers)
            response.raise_for_status() # يثير استثناء في حالة 4xx/5xx
            data = response.json()
            if data.get("success") == 1:
                # نُرجع قائمة بالعملات التي تحتوي على ID (جميعها)
                return [c for c in data.get("data", []) if c.get("currency_network_id")]
            else:
                # طباعة الخطأ إذا كان الرد ناجح HTTP لكن العملية فشلت
                print(f"❌ Arab Payment API Error (getCurrencies): {data.get('message')}")
                return []
    except Exception as e:
        # طباعة خطأ الاتصال
        print(f"❌ HTTP Request Error (getCurrencies): {e}")
        return []
# --- [بداية الإضافة: دالة إرسال الإشعارات] ---
async def send_deposit_notification(context: ContextTypes.DEFAULT_TYPE, user, status: str, amount: float, method: str, client_number: str, details: str = ""):
    # ❌ تم حذف: db = load_db()
    """
    إرسال إشعار بحالة الإيداع إلى قناة العمليات
    """
    # ⚠️ يفترض أن الثوابت (OPERATIONS_CHANNEL_ID و PAYMENT_METHODS) 
    # تم استيرادها بشكل صحيح في بداية ملفك
    if not OPERATIONS_CHANNEL_ID:
        print("⚠️ OPERATIONS_CHANNEL_ID is not set. Skipping notification.")
        return

    user_mention = user.mention_html()
    username = f"@{user.username}" if user.username else "N/A"
    # يتم استخدام المتغير method_name كما هو في الكود الأصلي، 
    # مع افتراض أن PAYMENT_METHODS هو قاموس مُستورد
    method_name = PAYMENT_METHODS.get(method, method)
    
    if status == "SUCCESS":
        icon = "✅"
        title = "إيداع ناجح (آلي)"
        message = (
            f"{icon} <b>{title}</b>\n\n"
            f"👤 <b>المستخدم:</b> {user_mention} (<code>{user.id}</code>)\n"
            f"📧 <b>المعرف:</b> {username}\n"
            f"💰 <b>المبلغ:</b> <code>{int(amount):,}</code> ل.س\n"
            f"💳 <b>الطريقة:</b> {method_name}\n"
            f"🧾 <b>رقم العملية:</b> <code>{client_number}</code>"
        )
    elif status == "FAILED":
        icon = "❌"
        title = "فشل إيداع (آلي)"
        message = (
            f"{icon} <b>{title}</b>\n\n"
            f"👤 <b>المستخدم:</b> {user_mention} (<code>{user.id}</code>)\n"
            f"📧 <b>المعرف:</b> {username}\n"
            f"💳 <b>الطريقة:</b> {method_name}\n"
            f"🧾 <b>الرقم المدخل:</b> <code>{client_number}</code>\n"
            f"🚫 <b>السبب:</b> {details}"
        )
    elif status == "DUPLICATE":
        icon = "⚠️"
        title = "محاولة تكرار عملية إيداع"
        message = (
            f"{icon} <b>{title}</b>\n\n"
            f"👤 <b>المستخدم:</b> {user_mention} (<code>{user.id}</code>)\n"
            f"📧 <b>المعرف:</b> {username}\n"
            f"💳 <b>الطريقة:</b> {method_name}\n"
            f"🧾 <b>الرقم المكرر:</b> <code>{client_number}</code>"
        )
    elif status == "FRAUD":
        icon = "🚨"
        title = "محاولة احتيال مكتشفة"
        message = (
            f"{icon} <b>{title}</b>\n\n"
            f"👤 <b>المستخدم:</b> {user_mention} (<code>{user.id}</code>)\n"
            f"💳 <b>الطريقة:</b> {method_name}\n"
            f"🧾 <b>الرقم المشبوه:</b> <code>{client_number}</code>\n"
            f"⚠️ <b>التفاصيل:</b> {details}"
        )

    else:
        return # لا ترسل إشعار لحالات أخرى مثل حوالة قيد الانتظار

    try:
        await context.bot.send_message(
            chat_id=OPERATIONS_CHANNEL_ID,
            text=message,
            parse_mode='HTML'
        )
    except Exception as e:
        print(f"❌ خطأ في إرسال إشعار للقناة: {e}")

# --- [نهاية الإضافة] ---

def get_cancel_button():
    
    return [InlineKeyboardButton("❌ إلغاء العملية", callback_data="cancel_deposit")]


async def cancel_existing_operation(update, context, message="❌ تم إلغاء العملية السابقة تلقائيًا وبدء عملية جديدة."):
    
    if context.user_data.get("is_withdrawing") or context.user_data.get("deposit_method") or context.user_data.get("deposit_amount"):
        if update.callback_query:
            await update.callback_query.edit_message_text(message, parse_mode='HTML')
        elif update.message:
            await update.message.reply_text(message, parse_mode='HTML')
        context.user_data.clear()


async def deposit_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    
    query = update.callback_query
    await query.answer()

    if not is_deposit_withdraw_enabled():
        query = update.callback_query
        await query.answer()
        await query.edit_message_text("⛔ عمليات الشحن حالياً متوقفة من قبل الأدمن.")
        return ConversationHandler.END

    if context.user_data.get("is_withdrawing") or context.user_data.get("deposit_method"):
        await cancel_existing_operation(update, context, message="❌ تم إلغاء العملية السابقة تلقائيًا. الرجاء إعادة الضغط على شحن البوت للبدء من جديد.")
        return ConversationHandler.END

    context.user_data['is_depositing'] = True

    keyboard = [
    [
        InlineKeyboardButton("Syriatel Cash", callback_data="dep_method_syriatel_cash"),
        InlineKeyboardButton("Sham Cash AUTO(USD , SYP)", callback_data="dep_method_cham_cash"),
        InlineKeyboardButton( "💳syriatel إيداع يدوي", callback_data="manual_deposit")

    ],
    [
        InlineKeyboardButton("USDT", callback_data="dep_method_usdt"),
    ],
    [
        InlineKeyboardButton("🔙 العودة", callback_data="back_to_main"),
        InlineKeyboardButton("❌ إلغاء العملية", callback_data="cancel_deposit")
    ]
]

    await query.edit_message_text(
        "💰 <b>شحن المحفظة</b>\n\nالرجاء اختيار طريقة الإيداع:" + CANCEL_NOTE,
        reply_markup=InlineKeyboardMarkup(keyboard),
        parse_mode='HTML'
    )

    return AWAITING_DEPOSIT_METHOD



async def deposit_usdt_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ تم حذف: db = load_db() 
    query = update.callback_query
    await query.answer()

    # 1. التحقق من حالة الشحن والسحب (يجب أن تستخدم هذه الدالة _get_setting)
    if not is_deposit_withdraw_enabled():
        await query.edit_message_text("⛔ عمليات الشحن حالياً متوقفة من قبل الأدمن.")
        return ConversationHandler.END

    if context.user_data.get('deposit_method') or context.user_data.get('deposit_amount'):
        await query.edit_message_text("❌ تم إلغاء العملية السابقة تلقائيًا بسبب الضغط على زر جديد.", parse_mode='HTML')
        context.user_data.clear()
        return ConversationHandler.END

    user_id_str = str(query.from_user.id)
    
    # 2. التحقق من وجود الحساب باستخدام الدالة الجديدة
    # 🟢 get_user_data() ترجع قاموس البيانات أو قاموس فارغ {} إذا لم يكن موجودًا
    account_data = get_user_data(user_id_str) 
    
    if not account_data:
        await query.edit_message_text("❌ ليس لديك حساب بعد. الرجاء إنشاء حساب أولًا قبل الإيداع.", parse_mode='HTML')
        return ConversationHandler.END

    wallets = DEPOSIT_INFO.get("usdt", [])
    if not wallets:
        await query.edit_message_text("❌ لم يتم إعداد عنوان محفظة USDT.", parse_mode='HTML')
        return ConversationHandler.END

    context.user_data['deposit_method'] = "usdt"

    wallets_text = ""
    for idx, w in enumerate(wallets, start=1):
        addr = w.get("address") if isinstance(w, dict) else w
        wtype = w.get("type", "TRC20") if isinstance(w, dict) else "TRC20"
        wallets_text += f"{idx}. {wtype}: <code>{addr}</code>\n"

    cancel_button = get_cancel_button()
    await query.edit_message_text(
        f"💰 لإيداع USDT:\n\n➡️ أرسل المبلغ بالـ USDT الذي تريد إيداعه.\n\nمحفظة الإيداع:\n{wallets_text}" + CANCEL_NOTE,
        parse_mode="HTML",
        reply_markup=InlineKeyboardMarkup([cancel_button])
    )
    return AWAITING_DEPOSIT_AMOUNT
async def receive_deposit_method(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ تم حذف: db = load_db()
    # ❌ تم حذف: from database import db
    
    query = update.callback_query
    await query.answer()
    data = query.data

    if data == "cancel_deposit":
        from handlers.menu.main_menu import start
        await query.edit_message_text("❌ تم إلغاء عملية الإيداع.", parse_mode='HTML')
        context.user_data.clear()
        return ConversationHandler.END

    # 1. معالجة اختيار عملة محددة من Arab Payment (مثل usdt_arab_pay_id_123)
    if data.startswith("usdt_arab_pay_id_"):
        currency_id = int(data.split("_")[-1])
        available_currencies = context.user_data.get('arab_pay_currencies', [])
        selected = next((c for c in available_currencies if c['currency_network_id'] == currency_id), None)
        if not selected:
            await query.edit_message_text("❌ العملة غير موجودة في القائمة.", parse_mode='HTML')
            return ConversationHandler.END

        context.user_data['deposit_method'] = 'usdt'
        context.user_data['arab_pay_currency_id'] = currency_id
        context.user_data['currency_display'] = f"{selected['currency']}/{selected['network']}"
        context.user_data['selected_currency'] = selected

        cfg = get_config()
        min_deposit = cfg.get("MINIMUM_DEPOSIT", {}).get("USDT", 1.0)
        currency_label = selected['currency']

        text = (
            f"💰 اخترت الإيداع عبر <b>{context.user_data['currency_display']}</b>.\n\n"
            f"🔸 <b>الحد الأدنى للإيداع:</b> <code>{min_deposit} {currency_label}</code>\n\n"
            f"📩 الرجاء إدخال المبلغ الذي تريد إيداعه بـ {currency_label}."
        )
        await query.edit_message_text(
            text + CANCEL_NOTE,
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup([get_cancel_button()])
        )
        return AWAITING_DEPOSIT_AMOUNT

    # 2. معالجة الضغط على "USDT" من القائمة الرئيسية → جلب العملات
    if data == "dep_method_usdt":
        await query.edit_message_text("⏳ جاري جلب طرق الدفع المتاحة من Luffy Bot...", parse_mode='HTML')
        available_currencies = await get_arab_payment_currencies()
        if not available_currencies:
            await query.edit_message_text(
                "❌ لا يوجد عملات متاحة حالياً عبر Arab Payment. يرجى المحاولة لاحقاً.",
                parse_mode='HTML'
            )
            return ConversationHandler.END

        context.user_data['arab_pay_currencies'] = available_currencies
        keyboard_buttons = []
        for currency_info in available_currencies:
            btn_text = f"{currency_info['currency']} ({currency_info['network']})"
            callback_data = f"usdt_arab_pay_id_{currency_info['currency_network_id']}"
            keyboard_buttons.append(InlineKeyboardButton(btn_text, callback_data=callback_data))

        keyboard = [keyboard_buttons[i:i + 2] for i in range(0, len(keyboard_buttons), 2)]
        keyboard.append(get_cancel_button())

        await query.edit_message_text(
            "💰 اختر العملة والشبكة لإتمام الإيداع:",
            reply_markup=InlineKeyboardMarkup(keyboard),
            parse_mode='HTML'
        )
        return AWAITING_DEPOSIT_METHOD

    # 3. معالجة شام كاش → عرض خيارات العملة
    if data == "dep_method_cham_cash":
        keyboard = [
            [InlineKeyboardButton("💵 إيداع بالدولار", callback_data="cham_cash_usd")],
            [InlineKeyboardButton("💴 إيداع بالليرة السورية", callback_data="cham_cash_syr")],
            [InlineKeyboardButton("❌ إلغاء العملية", callback_data="cancel_deposit")]
        ]
        await query.edit_message_text(
            "💰 اختر نوع الإيداع لشام كاش:",
            reply_markup=InlineKeyboardMarkup(keyboard),
            parse_mode='HTML'
        )
        return AWAITING_DEPOSIT_METHOD

    # 4. معالجة شام كاش USD / SYP (بعد الاختيار)
    if data in ["cham_cash_usd", "cham_cash_syr"]:
        await cancel_existing_operation(update, context)
        context.user_data['deposit_method'] = "cham_cash"
        context.user_data['cham_currency'] = "USD" if data == "cham_cash_usd" else "SYP"

        cfg = get_config()
        min_deposit_syp = cfg.get("MINIMUM_DEPOSIT", {}).get("SHAM", 10000)
        
        # 🟢 التعديل الرئيسي: استخدام دالة _get_setting لجلب سعر الصرف
        exchange_rate = _get_setting('exchange_rate', 4500)

        if context.user_data['cham_currency'] == "USD":
            # يجب تحويل سعر الصرف إلى float للتأكد من عملية القسمة
            try:
                 exchange_rate = float(exchange_rate)
            except ValueError:
                 exchange_rate = 4500.0 # قيمة احتياطية

            min_usd = round(min_deposit_syp / exchange_rate, 2)
            text = (
                f"💰 اخترت الإيداع عبر <b>شام كاش بالدولار الأمريكي (USD)</b>.\n\n"
                f"💵 <b>سعر الصرف الحالي:</b> <code>{exchange_rate:,.0f} ل.س / 1 USD</code>\n"
                f"🔸 <b>الحد الأدنى للإيداع:</b> <code>{min_usd} USD</code>\n\n"
                f"📩 الرجاء إدخال المبلغ الذي تريد إيداعه بالدولار."
            )
        else:
            text = (
                "💴 اخترت الإيداع عبر <b>شام كاش بالليرة السورية</b>.\n\n"
                f"🔸 <b>الحد الأدنى للإيداع:</b> <code>{min_deposit_syp:,} ل.س</code>\n\n"
                "📩 الرجاء إدخال المبلغ الذي تريد إيداعه بالليرة."
            )

        await query.edit_message_text(
            text + CANCEL_NOTE,
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup([get_cancel_button()])
        )
        return AWAITING_DEPOSIT_AMOUNT

    # 5. باقي الطرق: syriatel_cash, hawala
    method_key_map = {
        "dep_method_syriatel_cash": "syriatel_cash",
        "dep_method_hawala": "hawala"
    }

    if data in method_key_map:
        method_key = method_key_map[data]
        context.user_data['deposit_method'] = method_key

        cfg = get_config()
        min_deposit_config = cfg.get("MINIMUM_DEPOSIT", {})
        key_map = {"syriatel_cash": "SYRIATEL", "hawala": "HAWALA"}
        config_key = key_map[method_key]
        min_deposit = min_deposit_config.get(config_key, 0)
        currency_display = "ل.س"
        method_name = PAYMENT_METHODS.get(method_key, method_key)
        if not is_deposit_method_enabled("deposit_syriatel_cash"):
            await query.edit_message_text(
            "⛔️ <b>الإيداع عبر سيرياتيل كاش متوقف حالياً.</b>\n\n"
            "📩 يرجى اختيار طريقة إيداع أخرى، أو التواصل مع الدعم للمساعدة.",
            parse_mode='HTML',
            
        )

            return ConversationHandler.END
        await query.edit_message_text(
            f"💰 لقد اخترت <b>{method_name}</b> كطريقة للإيداع.\n\n"
            f"🔸 <b>الحد الأدنى للإيداع:</b> <code>{min_deposit:,} {currency_display}</code>\n\n"
            f"📩 الرجاء الآن إدخال المبلغ الذي تريد إيداعه:"
            + CANCEL_NOTE,
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup([get_cancel_button()])
        )
        return AWAITING_DEPOSIT_AMOUNT
    await query.edit_message_text("❌ خيار غير معروف. الرجاء العودة والمحاولة من جديد.", parse_mode='HTML')
    return ConversationHandler.END

async def receive_deposit_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ تم حذف: db = load_db()
    # ❌ تم حذف: from database import db
    
    method_key = context.user_data.get('deposit_method')
    cfg = get_config()
    min_deposit_config = cfg.get("MINIMUM_DEPOSIT", {})
    user = update.effective_user
    if not method_key:
        await update.message.reply_text("❌ انتهت صلاحية العملية. يرجى البدء من قائمة 'شحن المحفظة'.", parse_mode='HTML')
        context.user_data.clear()
        return ConversationHandler.END
    
    # 🧭 خريطة بين أسماء الطرق والمفاتيح داخل config
    key_map = {
        "syriatel_cash": "SYRIATEL",
        "cham_cash": "SHAM",
        "usdt": "USDT",
        "hawala": "HAWALA"
    }

    config_key = key_map.get(method_key, method_key.upper())
    min_deposit = min_deposit_config.get(config_key, 0)

    # 🔢 محاولة قراءة المبلغ المدخل من المستخدم
    try:
        amount = float(update.message.text)
    except (ValueError, TypeError):
        await update.message.reply_text(
            "❌ المبلغ غير صالح. الرجاء إدخال رقم صحيح.",
            parse_mode='HTML'
        )
        return AWAITING_DEPOSIT_AMOUNT

    context.user_data['deposit_amount'] = amount  # حفظ المبلغ
    
    # 🔒 التحقق من الحد الأدنى لكل طريقة
    if method_key == "cham_cash":
        
        # 🟢 التعديل الرئيسي: استخدام دالة _get_setting لجلب سعر الصرف
        exchange_rate_str = _get_setting("exchange_rate", 4500)
        try:
             exchange_rate = float(exchange_rate_str)
        except ValueError:
             exchange_rate = 4500.0 # قيمة احتياطية

        selected_currency = context.user_data.get("cham_currency", "SYP")

        if selected_currency == "USD" and exchange_rate > 0:
            effective_min_deposit = round(min_deposit / exchange_rate, 2)
            currency_display = "USD"
        else:
            effective_min_deposit = min_deposit
            currency_display = "ل.س"
    elif method_key == "usdt":
        effective_min_deposit = min_deposit
        currency_display = "USDT"
    else:
        effective_min_deposit = min_deposit
        currency_display = "ل.س"

    if amount < effective_min_deposit:
        await update.message.reply_text(
            f"❌ الحد الأدنى للإيداع لطريقة {PAYMENT_METHODS.get(method_key, method_key)} هو {effective_min_deposit:,} {currency_display}",
            parse_mode='HTML'
        )
        return AWAITING_DEPOSIT_AMOUNT

    cancel_button = get_cancel_button()

    # 🪙 مسار USDT / Arab Payment
    if method_key == 'usdt':
        currency_id = context.user_data.get('arab_pay_currency_id')
        currency_display = context.user_data.get('currency_display')

        if not currency_id:
            await update.message.reply_text(
                "❌ خطأ داخلي: لم يتم تحديد العملة. يرجى البدء من جديد.",
                parse_mode='HTML'
            )
            context.user_data.clear()
            return ConversationHandler.END

        await update.message.reply_text(
            "⏳ جاري إنشاء فاتورة الدفع، يرجى الانتظار...",
            parse_mode='HTML'
        )

        url = f"{API_BASE_URL}/createPayment"
        headers = {
            "Authorization": f"Bearer {ARAB_PAYMENT_API_KEY}",
            "Content-Type": "application/json"
        }
        payload = {
            "currency_network_id": currency_id,
            "amount": amount,
            "email": f"{user.username}@example.com" if user.username else f"user_{user.id}@example.com",
            "data_1": str(user.id),
            "data_2": currency_display
        }
        

        try:
            async with httpx.AsyncClient(timeout=15) as client:
                response = await client.post(url, headers=headers, json=payload)
                response.raise_for_status()
                result = response.json()

                if result.get("success") == 1 and result.get("data"):
                    payment_info = result['data'][0]
                    transaction_id = payment_info['transaction_id']
                    context.user_data['arab_pay_transaction_id'] = transaction_id

                    msg = (
                        f"✅ تم إنشاء فاتورة دفع بنجاح.\n\n"
                        f"💰 <b>المبلغ المطلوب:</b> <code>{amount} {currency_display.split('/')[0]}</code>\n"
                        f"💳 <b>العنوان:</b> <code>{payment_info['address']}</code>\n"
                        f"⏰ <b>ينتهي في:</b> <code>{payment_info['stop_at'].split(' ')[1]}</code>\n"
                        f"🔗 <b>رابط الدفع:</b> {payment_info['payment_link']}\n\n"
                        f"<b>قم بالدفع الآن، ثم اضغط على 'تحقق من الدفع'.</b>"
                        f"\n\n🚨 <b>مهم:</b> في حال عدم وصول المبلغ، يجب استخدام زر 'تحقق من الدفع'."
                        + CANCEL_NOTE
                    )

                    keyboard = [
                        [InlineKeyboardButton("✅ تحقق من حالة الدفع", callback_data=f"check_arab_pay:{transaction_id}")],
                        cancel_button
                    ]

                    await update.message.reply_text(
                        msg,
                        parse_mode='HTML',
                        reply_markup=InlineKeyboardMarkup(keyboard)
                    )
                    return AWAITING_DEPOSIT_PROOF
                else:
                    error_msg = result.get("message", "خطأ غير معروف من Arab Payment.")
                    await update.message.reply_text(f"❌ فشل في إنشاء الفاتورة: {error_msg}", parse_mode='HTML')
                    context.user_data.clear()
                    return ConversationHandler.END

        except httpx.HTTPStatusError as e:
            await update.message.reply_text(f"❌ خطأ في الاتصال بالـ API: {e}", parse_mode='HTML')
            context.user_data.clear()
            return ConversationHandler.END
        except Exception as e:
            await update.message.reply_text(f"❌ خطأ فني غير متوقع: {e}", parse_mode='HTML')
            context.user_data.clear()
            return ConversationHandler.END

    # 💸 باقي طرق الإيداع (شام كاش، سيريتيل، حوالة...)
    else:
        deposit_details = DEPOSIT_INFO.get(method_key, "")
        method_name = PAYMENT_METHODS.get(method_key, method_key)
        proof_request = "📷 صورة إثبات التحويل (الإيصال)" if method_key == 'hawala' else "رقم عملية التحويل"

        if method_key == "syriatel_cash" and isinstance(deposit_details, list):
            # 🔥 عرض الأرقام الظاهرة فقط
            visible_numbers = [
                item["number"]
                for item in deposit_details
                if item.get("visible", True)
            ]
            deposit_details_formatted = "\n".join(
                f"{idx}. <code>{num}</code>"
                for idx, num in enumerate(visible_numbers, start=1)
            )
        elif isinstance(deposit_details, list):
            deposit_details_formatted = "\n".join(
                f"{idx}. <code>{num}</code>" for idx, num in enumerate(deposit_details, start=1)
            )
        elif isinstance(deposit_details, str):
            deposit_details_formatted = f"<code>{deposit_details}</code>"
        else:
            deposit_details_formatted = str(deposit_details)

        currency_label = "USD" if method_key == "cham_cash" and context.user_data.get("cham_currency") == "USD" else "ل.س"

        await update.message.reply_text(
            f"✅ لإتمام إيداع مبلغ <code>{amount}</code> {currency_label}:\n\n"
            f"1. قم بالتحويل عبر <b>{method_name}</b> إلى:\n{deposit_details_formatted}\n\n"
            f"2. بعد التحويل أرسل <b>{proof_request}</b> هنا."
            + CANCEL_NOTE,
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup([cancel_button])
        )
        return AWAITING_DEPOSIT_PROOF
# --- [بداية التعديل الجذري] ---
async def receive_deposit_proof(update: Update, context: ContextTypes.DEFAULT_TYPE):
    
    # ❌ تم حذف: from database import db
    # ❌ تم حذف: db = load_db()
    
    from handlers.deposit.usdt import check_arab_payment_status # أبقيت على هذا الاستيراد إذا كان داخلياً
    user = update.effective_user
    user_id_str = str(user.id)
    amount_input = context.user_data.get('deposit_amount')
    method_key = context.user_data.get('deposit_method')
    
    # 1. Validation
    if not amount_input or not method_key:
        # التأكد من الرد في حالتي الرسالة أو الكولباك
        if update.message:
            await update.message.reply_text("❌ حدث خطأ، الرجاء إعادة العملية من البداية.", parse_mode='HTML')
        elif update.callback_query:
            await update.callback_query.edit_message_text("❌ حدث خطأ، الرجاء إعادة العملية من البداية.", parse_mode='HTML')
        context.user_data.clear()
        return ConversationHandler.END

    # تهيئة المتغيرات
    client_number = update.message.text.strip() if update.message and update.message.text else None
    proof_photo_id = update.message.photo[-1].file_id if update.message and update.message.photo else None
    
    cfg = get_config()
    bonus_key_map = {
        "usdt": "USDT", "cham_cash": "SHAM", "syriatel_cash": "SYRIATEL", "hawala": "HAWALA"
    }
    bonus_percent = float(cfg.get("Bouns_Gift", {}).get(bonus_key_map.get(method_key, method_key.upper()), 0))

    verification_status = "PENDING"
    amount_ls = 0.0
    error_details = ""
    tx_or_client_number = client_number or "N/A" # ID المرجع لعملية التسجيل والإشعارات

    # --- 🛑 1. معالجة Callback زر التحقق من Arab Payment (الحالة الأساسية لـ USDT) ---
    if update.callback_query and update.callback_query.data.startswith("check_arab_pay:"):
        
        query = update.callback_query
        await query.answer("جاري التحقق من حالة الدفع...")
        
        transaction_id = query.data.split(':')[-1]
        tx_or_client_number = transaction_id # نستخدم ID العملية كمرجع
        
        # التأكد من أن العملية تخص المستخدم الحالي ونظام Arab Payment
        if context.user_data.get('deposit_method') != 'usdt' or context.user_data.get('arab_pay_transaction_id') != transaction_id:
             await query.edit_message_text("❌ لم يتم العثور على تفاصيل هذه العملية في محادثتك الحالية.", parse_mode='HTML')
             context.user_data.clear()
             return ConversationHandler.END

        # 🏃‍♂️ إجراء التحقق من API
        status_code, payment_details = await check_arab_payment_status(context, transaction_id) 
        
        if status_code == 1: # 🟢 تم الاستلام (Received)
            
            # 🟢 التعديل: استخدام دالة check_used_transaction الجديدة
            if await check_used_transaction(transaction_id):
                verification_status = "ALREADY_USED"
                error_details = "تم استخدام هذه العملية سابقاً."
            else:
                verification_status = "SUCCESS"
                
                # 🟢 التعديل: جلب سعر الصرف من الإعدادات
                exchange_rate_str = _get_setting('exchange_rate', 4500)
                try:
                    exchange_rate = float(exchange_rate_str)
                except ValueError:
                    exchange_rate = 4500.0
                
                # المبلغ المستلم بالعملة الرقمية (قد يختلف قليلاً عن المبلغ المطلوب)
                received_amount_crypto = float(payment_details.get("received_amount", 0.0)) 
                amount_ls = received_amount_crypto * exchange_rate
                
                # 🟢 التعديل: حفظ العملية كمعاملة مستخدمة في قاعدة البيانات
                await add_used_transaction(transaction_id, amount_ls, 'usdt_arab_pay', user_id_str)
                
                await query.edit_message_text(
                    f"✅ تم استلام العملية بنجاح!\n"
                    f"💰 <b>المبلغ المستلم:</b> <code>{received_amount_crypto}</code> USDT\n"
                    f"💵 <b>يعادل:</b> <code>{int(amount_ls):,}</code> ل.س\n\n"
                    f"جارٍ إضافة الرصيد إلى محفظتك...",
                    parse_mode='HTML'
                )
                
                # إشعار سريع للانتهاء (سيتم تعديل الرسالة لاحقاً في قسم SUCCESS)
                await query.edit_message_text("✅ تم التحقق بنجاح! جاري إضافة الرصيد...", parse_mode='HTML')
            
        elif status_code == 0: # ⏳ Waiting
            await query.edit_message_text("⏳ العملية لا تزال قيد الانتظار في الشبكة. يرجى المحاولة لاحقاً.", parse_mode='HTML')
            return AWAITING_DEPOSIT_PROOF 
            
        elif status_code == 2: # ❌ TimeOut
            verification_status = "NOT_FOUND" 
            error_details = "انتهى وقت الفاتورة المخصص للدفع."
            
        elif status_code == 3: # ⚠️ WaitingForMoreFunds
            await query.edit_message_text("⚠️ تم استلام جزء من المبلغ. يرجى إرسال المبلغ المتبقي لإتمام العملية.", parse_mode='HTML')
            return AWAITING_DEPOSIT_PROOF
        
        elif status_code < 0: # ❌ خطأ API أو اتصال
            verification_status = "ERROR"
            error_details = payment_details.get('error', "خطأ غير معروف في التحقق.")

    # --- 🛑 2. معالجة إدخال نص/صورة (لطرق Syriatel, Cham Cash, Hawala) ---
    elif update.message:
        
        # ❌ منع إدخال أي شيء يدوي لـ USDT (لأننا نستخدم Arab Pay)
        if method_key == "usdt":
            await update.message.reply_text("❌ يرجى عدم إرسال أي نصوص أو صور يدوياً. يجب عليك استخدام زر '✅ تحقق من حالة الدفع' للتحقق من الفاتورة.", parse_mode='HTML')
            return AWAITING_DEPOSIT_PROOF
            
        # 🟢 Syriatel Cash
        elif method_key == "syriatel_cash":
            if not client_number:
                await update.message.reply_text("❌ لم ترسل رقم العملية. الرجاء إرسال الرقم.", parse_mode='HTML')
                return AWAITING_DEPOSIT_PROOF

            # 🟢 التعديل: استخدام دالة check_used_transaction الجديدة
            if await check_used_transaction(client_number):
                verification_status = "ALREADY_USED"
            else:
                # ملاحظة: تم افتراض وجود دالة check_transaction
                verified = check_transaction(client_number, amount_input)

                if verified:
                    if verified.get("fraud"):
                        verification_status = "FRAUD"
                        error_details = "محاولة احتيال مكتشفة (status=0)"
                    else:
                        verification_status = "SUCCESS"
                        amount_ls = float(amount_input)
                        try:
                            from handlers.referral.referral_service import process_referral_for_deposit
                            deposit_tx = {
                                "transaction_id": client_number,
                                "amount": amount_ls * 100
                            }
                            await process_referral_for_deposit(
                                invited_id=user_id_str,
                                deposit_tx=deposit_tx,
                                bot=context.bot
                            )
                            print(f"✅ تمت معالجة الإحالة لـ {user_id_str}")
                        except Exception as e:
                            print(f"⚠️ فشل معالجة الإحالة (Syriatel Cash): {e}")
                else:
                    verification_status = "NOT_FOUND"
                    error_details = "لم يتم العثور على العملية / المبلغ غير مطابق"

            

        
        # elif method_key == "syriatel_cash":
        #     if not client_number:
        #         await update.message.reply_text(
        #             "❌ لم ترسل رقم العملية. الرجاء إرسال الرقم.",
        #             parse_mode="HTML"
        #         )
        #         return AWAITING_DEPOSIT_PROOF

        #     # محاولة إضافة الرصيد والتحقق
        #     success = await credit_wallet(
        #         transaction_no=client_number,
        #         amount=int(amount_input),
        #         user_id=update.effective_user.id,
        #         bot=context.bot
        #     )

        #     # تحديد الحالة لتقرير نوع الرسالة والقناة
        #     if success:
        #         status = "SUCCESS"
        #         details = ""
        #         user_message = (f"✅ تم تأكيد إيداعك بنجاح!\n\n"
        #                     f"💰 المبلغ الأساسي: <code>{int(amount_input):,}</code> ل.س\n")
        #     else:
        #         already_used = await check_used_transaction(client_number)
        #         trx = check_transaction(client_number, int(amount_input))

        #         if already_used:
        #             status = "DUPLICATE"
        #             details = ""
        #             user_message = "⚠️ العملية تم استخدامها سابقاً."
        #         elif trx and trx.get("fraud"):
        #             status = "FRAUD"
        #             details = "محاولة احتيال مكتشفة (status=0)"
        #             user_message = "🚨 العملية مشبوهة!"
        #         else:
        #             status = "FAILED"
        #             details = "لم يتم العثور على العملية أو المبلغ غير مطابق"
        #             user_message = "❌ فشل التحقق من العملية أو تم استخدامها سابقاً."

        #     # إرسال رسالة للمستخدم
        #     await update.message.reply_text(
        #         user_message,
        #         parse_mode="HTML"
        #     )

        #     # إرسال إشعار للقناة عبر الدالة الموحدة
        #     await send_deposit_notification(
        #         context=context,
        #         user=update.effective_user,
        #         status=status,
        #         amount=float(amount_input),
        #         method="syriatel",
        #         client_number=client_number,
        #         details=details
        #     )

        #     return ConversationHandler.END



            
        # 🟢 Cham Cash
        elif method_key == "cham_cash":
            if not client_number:
                await update.message.reply_text("❌ لم ترسل رقم العملية. الرجاء إرسال الرقم.", parse_mode='HTML')
                return AWAITING_DEPOSIT_PROOF
            
            # 1. إرسال رسالة الانتظار فوراً
            await update.message.reply_text("⏳ يرجى الانتظار لحظة جارٍ التأكد من العملية  ...", parse_mode='HTML')

            # 2. تحضير متغيرات الثريد
            currency = "USD" if context.user_data.get("cham_currency") == "USD" else "SYP"
            main_loop = asyncio.get_running_loop()

            # 3. تشغيل الدالة المعدلة التي تستخدم API فقط
            Thread(
                target=run_auto_deposit_check,
                args=(
                    user_id_str, 
                    client_number, 
                    amount_input, 
                    currency, 
                    context.bot, 
                    context, 
                    update.effective_user, 
                    main_loop 
                ),
                daemon=True
            ).start()
            
            # 4. إنهاء خطوة المحادثة فوراً للحفاظ على استجابة البوت
            return ConversationHandler.END

        # 🟡 Hawala (Manual) - Requires admin review
        elif method_key == "hawala":
            if not client_number and not proof_photo_id:
                await update.message.reply_text("❌ الرجاء إرسال رقم إيصال التحويل أو صورة الإيصال.", parse_mode='HTML')
                return AWAITING_DEPOSIT_PROOF

            verification_status = "MANUAL_REVIEW"
            await update.message.reply_text(
                "✅ تم استلام إثبات التحويل. سيتم مراجعة طلبك وإضافة الرصيد يدوياً في أقرب وقت ممكن.",
                parse_mode='HTML'
            )
            # هنا يجب إرسال إشعار للمسؤول للمراجعة اليدوية (يتم إرساله لاحقاً في قسم الإشعارات)
            # *لا ينهي المحادثة هنا لإرسال الإشعار* # ❌ مدخل غير متوقع
        else:
              await update.message.reply_text("❌ الرجاء إكمال عملية الإيداع عبر زر 'تحقق من حالة الدفع'.", parse_mode='HTML')
              context.user_data.clear()
              return ConversationHandler.END


    # --- 🛑 3. لا يوجد مدخل صالح (لا رسالة ولا كولباك) ---
    else:
        # يمكن أن يحدث إذا لم يضغط المستخدم الزر أو كانت هناك مشكلة غير متوقعة
        if update.effective_message:
            await update.effective_message.reply_text("❌ الرجاء إكمال عملية الإيداع.", parse_mode='HTML')
        context.user_data.clear()
        return ConversationHandler.END


    # --- [النهاية: التعامل مع النتائج (مشترك بين كل الطرق)] ---

    # 1. إذا نجح الإيداع
    if verification_status == "SUCCESS":
        # 🟢 إضافة أرباح الإحالة (فقط عند النجاح)
        if method_key == "syriatel_cash":
            amount_ls = amount_ls * 100
        
        # 💾 تحديث الرصيد والسجلات (لـ USDT و Syriatel Cash)
        
        bonus_amount = round(amount_ls * (bonus_percent / 100)) if bonus_percent > 0 else 0
        total_amount = amount_ls + bonus_amount

        # 🟢 التعديل: تحديث رصيد المستخدم باستخدام الدالة الجديدة
        await update_user_balance(user_id_str, total_amount, amount_ls) 
        
        # 🟢 التعديل: تسجيل العملية باستخدام الدالة الجديدة
        await add_transaction_log(
            user_id_str, 'deposit', method_key.upper(),
            amount_ls, bonus_amount, tx_or_client_number
        )
        
        # 🟢 التعديل: تسجيل في سجلات الأدمن
        add_admin_history(
            user_id_str, "deposit", amount_ls, method_key.upper(), status="success", extra=f"client_num={tx_or_client_number}"
        )
        
        
        # 🟢 التعديل: تسجيل العملية كمستخدمة (لطرق الإيداع اليدوي/نصف آلي: Syriatel)
        if method_key == "syriatel_cash":
             await add_used_transaction(tx_or_client_number, amount_ls, method_key, user_id_str)
             
        
        # ❌ تم حذف: save_data(db)
        
        # 🟢 إرسال رسالة للمستخدم (يتم إرسالها فقط إذا لم يكن Cham Cash - الثريد يتولى إرسالها)
        if method_key != "cham_cash": 
            msg = (f"✅ تم شحن رصيدك بنجاح.\n"
                f"💰 المبلغ الأساسي: <code>{int(amount_ls):,}</code> ل.س\n")
            if bonus_amount > 0:
                msg += f"🎁 البونص: <code>{bonus_amount:,}</code> ل.س ({bonus_percent:.1f}%)\n"
            msg += f"💵 المجموع الكلي: <code>{int(total_amount):,}</code> ل.س"
            
            try:
                # نحاول التعديل إذا كان كولباك، وإلا نرد برسالة جديدة
                await update.effective_message.edit_text(msg, parse_mode='HTML')
            except Exception:
                await update.effective_message.reply_text(msg, parse_mode='HTML')

        # إرسال إشعار للقناة (مهم لجميع الطرق)
        await send_deposit_notification(context, user, "SUCCESS", amount_ls, method_key, tx_or_client_number)
        
    # 2. إذا كانت العملية مكررة
    elif verification_status == "ALREADY_USED":
        
        # 🟢 إرسال رسالة للمستخدم (لجميع الطرق باستثناء Cham Cash)
        if method_key != "cham_cash": 
            await update.effective_message.reply_text("❌ هذا الرقم تم استخدامه في عملية سابقة. لا يمكن استخدامه مرة أخرى.", parse_mode='HTML')
            
        await send_deposit_notification(context, user, "DUPLICATE", 0, method_key, tx_or_client_number)

    # 3. إذا لم يتم العثور عليها أو فشلت (آلياً)
    elif verification_status in ["NOT_FOUND", "MINIMUM_AMOUNT", "ERROR"]:
        
        # 🟢 إرسال رسالة للمستخدم (لجميع الطرق باستثناء Cham Cash)
        if method_key != "cham_cash": 
            user_message_map = {
                "NOT_FOUND": "❌ لم يتم العثور على عملية دفع بهذا الرقم، أو المبلغ غير مطابق.",
                "MINIMUM_AMOUNT": f"❌ المبلغ أقل من الحد الأدنى للإيداع.",
                "ERROR": "❌ حدث خطأ فني أثناء التحقق. يرجى مراجعة الدعم."
            }
            await update.effective_message.reply_text(user_message_map.get(verification_status), parse_mode='HTML')
            
        await send_deposit_notification(context, user, "FAILED", 0, method_key, tx_or_client_number, details=error_details)
        
    elif verification_status == "FRAUD":
        
        # 🟢 إرسال رسالة للمستخدم (لجميع الطرق باستثناء Cham Cash)
        if method_key != "cham_cash": 
            await update.effective_message.reply_text(
                "🚨 تم اكتشاف محاولة احتيال في العملية المرسلة.\n"
                "سيتم الإبلاغ عن هذه المحاولة وإيقاف المعاملة.",
                parse_mode='HTML'
            )
            
        await send_deposit_notification(
            context, user, "FRAUD", 0, method_key, tx_or_client_number,
            details=error_details
        )
        
    # 4. إذا كانت حوالة يدوية (MANUAL_REVIEW)
    elif verification_status == "MANUAL_REVIEW":
        # إرسال إشعار للمراجعة اليدوية للمسؤول
        await send_deposit_notification(
            context, user, "MANUAL_REVIEW", amount_input, method_key, 
            tx_or_client_number, details=f"صورة الإيصال: {proof_photo_id if proof_photo_id else 'لا توجد'} - الرقم: {client_number if client_number else 'لا يوجد'}"
        )
        # الرسالة أُرسلت بالفعل (لا حاجة للتعديل هنا)
    from handlers.menu.main_menu import show_main_menu
    await show_main_menu(update, context)

    context.user_data.clear()
    return ConversationHandler.END


async def cancel_deposit(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ تم حذف: db = load_db()
    from handlers.menu.main_menu import start
    
    if update.callback_query:
        query = update.callback_query
        await query.answer()
        await query.edit_message_text("❌ تم إلغاء عملية الإيداع.", parse_mode='HTML')
        # ⚠️ (ملاحظة: دالة start يجب أن تكون متوفرة)
    else:
        await update.message.reply_text("❌ تم إلغاء عملية الإيداع.", parse_mode='HTML')
        # ⚠️ (ملاحظة: دالة start يجب أن تكون متوفرة)
        
    # استدعاء start بعد الرد لضمان العودة للقائمة الرئيسية
    await start(update, context) 
    context.user_data.clear()
    return ConversationHandler.END