# handlers/gifting/gift_balance.py

from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
# 🟢 استيراد دوال SQLite الجديدة
from database import (
    get_user_data, 
    add_transaction, 
    add_user_history,
    update_user_field,
    _get_setting,
    _set_setting
    # ❌ تم حذف db, save_all_changes, load_db
)
from handlers.menu.main_menu import start
from constants import AWAITING_RECIPIENT_ID, AWAITING_GIFT_AMOUNT


async def gift_balance_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    يبدأ محادثة إهداء الرصيد.
    """
    query = update.callback_query
    await query.answer()
    user_id_str = str(query.from_user.id)

    # 🟢 التعديل: استخدام get_user_data
    sender_data = get_user_data(user_id_str)
    
    if not sender_data or sender_data.get('balance', 0.0) <= 0:
        keyboard = [[InlineKeyboardButton("🔙 العودة للقائمة الرئيسية", callback_data="back_to_main")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "🚫 **خطأ:** يجب أن يكون لديك رصيد في محفظتك لتتمكن من إهداء الآخرين.",
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        return ConversationHandler.END

    await query.edit_message_text(
        "🎁 لإهداء رصيد، الرجاء إرسال **معرّف تليغرام (ID)** الخاص بالمستخدم الذي تريد إرسال الهدية إليه.\n\n"
        "يمكنك كتابة /cancel في أي وقت لإلغاء العملية."
    )
    return AWAITING_RECIPIENT_ID


async def receive_recipient_id(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    يستلم ويتحقق من صحة معرّف المستلم.
    """
    keyboard = [[InlineKeyboardButton("🔙 العودة للقائمة الرئيسية", callback_data="back_to_main")]]
    reply_markup = InlineKeyboardMarkup(keyboard)
    sender_id_str = str(update.effective_user.id)
    recipient_id_str = update.message.text

    if not recipient_id_str.isdigit():
        await update.message.reply_text("❌ **معرّف غير صالح.** الرجاء إرسال ID رقمي صحيح.", reply_markup=reply_markup, parse_mode='Markdown')
        return ConversationHandler.END

    # 🟢 التعديل: استخدام get_user_data للتحقق من وجود المستخدم
    recipient_data = get_user_data(recipient_id_str)
    if not recipient_data:
        await update.message.reply_text("❌ **لم يتم العثور على المستخدم.** هذا المستخدم ليس لديه حساب في البوت بعد.", reply_markup=reply_markup, parse_mode='Markdown')
        return ConversationHandler.END
        
    if recipient_id_str == sender_id_str:
        await update.message.reply_text("❌ لا يمكنك إهداء رصيد لنفسك!", reply_markup=reply_markup, parse_mode='Markdown')
        return ConversationHandler.END

    context.user_data['recipient_id_str'] = recipient_id_str
    await update.message.reply_text("✅ **تم العثور على المستخدم.** الآن، الرجاء إرسال **المبلغ** الذي تريد إهداءه.")
    return AWAITING_GIFT_AMOUNT


async def receive_gift_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    يستلم المبلغ، ينفذ العملية، ويحفظ التغييرات.
    """
    sender_id_str = str(update.effective_user.id)
    recipient_id_str = context.user_data['recipient_id_str']
    
    keyboard = [[InlineKeyboardButton("🔙 العودة للقائمة الرئيسية", callback_data="back_to_main")]]
    reply_markup = InlineKeyboardMarkup(keyboard)

    try:
        amount_to_gift = float(update.message.text)
        if amount_to_gift <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ **مبلغ غير صالح.** الرجاء إرسال رقم موجب صحيح.", reply_markup=reply_markup, parse_mode='Markdown')
        return ConversationHandler.END

    # 🟢 التعديل: جلب بيانات المرسل (للحصول على الرصيد)
    sender_account = get_user_data(sender_id_str)
    
    # ⚠️ العمولة ثابتة 5%
    commission = amount_to_gift * 0.0
    total_cost = amount_to_gift + commission
    
    current_balance = sender_account.get('balance', 0.0)

    if current_balance < total_cost:
        error_msg = (
            f"🚫 **رصيدك غير كافٍ!**\n\n"
            f"المبلغ المراد إهداؤه: {amount_to_gift:,.0f}\n"
            f"عمولة التحويل (5%): {commission:,.0f}\n"
            f"**الإجمالي المطلوب:** {total_cost:,.0f}\n"
            f"**رصيدك الحالي:** {current_balance:,.0f}"
        )
        await update.message.reply_text(error_msg, reply_markup=reply_markup, parse_mode='Markdown')
        return ConversationHandler.END

    # --- تنفيذ التحويل باستخدام دوال SQLite ---
    
    # 1. خصم المبلغ من المرسل (Total Cost)
    # ⚠️ نستخدم دالة add_transaction مع type="withdraw" و amount=-total_cost لخصم الرصيد
    add_transaction(
        user_id=sender_id_str, 
        tx_type="gift_send", 
        amount=-total_cost, 
        method="gift", 
        status="success",
        client_number=recipient_id_str # لحفظ المستلم كمرجع
    )
    # تسجيل العملية في تاريخ المستخدم
    add_user_history(sender_id_str, "gift_send", -total_cost, f"gift_to_{recipient_id_str}")


    # 2. إضافة المبلغ للمستلم (Gift Amount)
    # ⚠️ نستخدم دالة add_transaction مع type="deposit" و amount=amount_to_gift لإضافة الرصيد
    add_transaction(
        user_id=recipient_id_str, 
        tx_type="gift_receive", 
        amount=amount_to_gift, 
        method="gift", 
        status="success",
        client_number=sender_id_str # لحفظ المرسل كمرجع
    )
    # تسجيل العملية في تاريخ المستخدم
    add_user_history(recipient_id_str, "gift_receive", amount_to_gift, f"gift_from_{sender_id_str}")


    # 3. تحديث أرباح الكاشير
    # 🟢 التعديل: جلب وتحديث أرباح الكاشير باستخدام _get_setting و _set_setting
    try:
        current_profit = float(_get_setting('cashier_profit', 0.0))
        new_profit = current_profit + commission
        _set_setting('cashier_profit', new_profit)
    except ValueError:
        # في حال لم يتمكن من تحويل الإعداد إلى رقم (احتياطي)
        _set_setting('cashier_profit', commission)


    # 4. إشعار المرسل
    # 🟢 التعديل: جلب الرصيد الجديد بعد التحديث
    new_sender_balance = get_user_data(sender_id_str).get('balance', 0.0)

    success_msg = (
        f"✅ **تمت عملية الإهداء بنجاح!**\n\n"
        f"لقد أرسلت `{amount_to_gift:,.0f}` ل.س إلى المستخدم `{recipient_id_str}`.\n"
        f"تم خصم عمولة بقيمة `{commission:,.0f}` ل.س.\n"
        f"رصيدك الجديد هو `{new_sender_balance:,.0f}` ل.س."
    )
    await update.message.reply_text(success_msg, reply_markup=reply_markup, parse_mode='Markdown')

    # 5. إشعار المستلم
    # 🟢 التعديل: جلب بيانات المرسل والمستلم بعد التحديث
    sender_info = get_user_data(sender_id_str).get('username', sender_id_str)
    new_recipient_balance = get_user_data(recipient_id_str).get('balance', 0.0)
    
    try:
        await context.bot.send_message(
            chat_id=int(recipient_id_str),
            text=(
                f"🎉 **لقد استلمت هدية!**\n\n"
                f"قام المستخدم `{sender_info}` بإهدائك مبلغ `{amount_to_gift:,.0f}` ل.س.\n"
                f"رصيدك الجديد هو `{new_recipient_balance:,.0f}` ل.س."
            ),
            parse_mode='Markdown'
        )
    except Exception as e:
        print(f"فشل إرسال إشعار للمستلم {recipient_id_str}. الخطأ: {e}")

    return ConversationHandler.END


async def cancel_gifting(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    تلغي عملية الإهداء وتعيد المستخدم للقائمة الرئيسية.
    """
    await update.message.reply_text("تم إلغاء عملية الإهداء.")
    await start(update, context)
    return ConversationHandler.END