# handlers/gifting/promo_user.py

from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
from database import (
    get_user_data, 
    update_user_field,
    get_promo_code_users,
    add_user_history,
    get_promo_code,
    has_user_used_promo_code,            # لجلب كود معين (منشور فقط)
    consume_promo_code,        # لتقليل عدد الاستخدامات
    get_all_active_promo_codes, # لعرض جميع الأكواد المنشورة
    add_transaction,           # لإضافة سجل المعاملة
    get_promo_code_unpublished # ✅ الجديدة: للتحقق من الأكواد غير المنشورة
) 
from constants import AWAITING_PROMO_CODE_INPUT

async def promo_menu_user(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    keyboard = [
        [InlineKeyboardButton("🎟️ تفعيل كود هدية", callback_data="activate_promo_code")],
        [InlineKeyboardButton("🔙 العودة للقائمة", callback_data="back_to_main")]
    ]
    reply_markup = InlineKeyboardMarkup(keyboard)
    await query.edit_message_text("🎁 اختر ما تريد:", reply_markup=reply_markup)


async def view_promo_offers(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    # 🟢 عرض الأكواد المنشورة فقط للمستخدمين
    promo_codes_list = get_all_active_promo_codes()
    
    if not promo_codes_list:
        text = "📭 لا توجد عروض أو أكواد هدايا متاحة حاليًا."
    else:
        lines = [f"🎟️ `{promo['code']}` → هدية بقيمة {int(promo['value']):,} ل.س" for promo in promo_codes_list]
        text = "🎁 العروض المتاحة للجميع:\n\n" + "\n".join(lines)

    keyboard = [[InlineKeyboardButton("🔙 العودة", callback_data="promo_menu_user")]]
    await query.edit_message_text(text, reply_markup=InlineKeyboardMarkup(keyboard), parse_mode='Markdown')


async def activate_promo_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    await query.message.reply_text(
        "🎟️ أدخل كود الهدية الذي حصلت عليه:\n\n"
        "💡 *ملاحظة:* يمكنك استخدام:\n"
        "• الأكواد المنشورة علناً\n"
        "• الأكواد الخاصة التي حصلت عليها من الأدمن"
    )
    return AWAITING_PROMO_CODE_INPUT
async def handle_promo_code_input(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user_id_str = str(update.effective_user.id)
    code = update.message.text.strip().upper()

    # 🟢 التعديل: البحث أولاً في الأكواد المنشورة
    promo_data = get_promo_code(code)
    
    # 🟢 إذا لم يكن منشوراً، البحث في الأكواد غير المنشورة
    if not promo_data:
        promo_data = get_promo_code_unpublished(code)
    
    if not promo_data:
        await update.message.reply_text(
            "❌ الكود غير صالح.\n"
            "قد يكون:\n"
            "• مكتوب بشكل خاطئ\n"
            "• تم استخدامه\n"
            "• غير مخصص لك"
        )
        return ConversationHandler.END

    # 🟢 التحقق مما إذا كان المستخدم استخدم الكود من قبل
    if has_user_used_promo_code(user_id_str, code):
        await update.message.reply_text(
            "❌ لقد استخدمت هذا الكود مسبقاً!\n"
            "👤 **القاعدة:** كل مستخدم يمكنه استخدام الكود مرة واحدة فقط"
        )
        return ConversationHandler.END

    # 🟢 التعديل: استخدام get_user_data للتحقق من وجود الحساب
    account = get_user_data(user_id_str)
    if not account:
        await update.message.reply_text("❌ يجب أن يكون لديك حساب لتفعيل الكود. اضغط على 'حسابي' لإنشاء واحد.")
        return ConversationHandler.END

    amount = promo_data.get('value', 0.0)
    uses_left = promo_data.get('uses_left', 0)
    is_published = promo_data.get('published', 0)
    
    # 1. التحقق من الاستخدامات المتاحة
    if uses_left <= 0:
        await update.message.reply_text("❌ انتهت صلاحية هذا الكود أو تم استخدامه بالكامل.")
        return ConversationHandler.END

    # التأكد من أن القيمة عددية قبل الإضافة
    try:
        amount = float(amount)
    except (TypeError, ValueError):
        await update.message.reply_text("❌ خطأ في قيمة الكود. يرجى إبلاغ الإدارة.")
        return ConversationHandler.END
        
    # --- منطق تطبيق الكود ---
    
    # 🟢 **التصحيح: إزالة تحديث الرصيد هنا (لأن add_transaction ستزيده)**
    # current_balance = account.get('balance', 0.0)
    # new_balance = current_balance + amount  # ❌ إزالة هذا السطر
    # update_user_field(user_id_str, "balance", new_balance)  # ❌ إزالة هذا السطر
    
    # 2. استهلاك استخدام واحد من الكود مع تسجيل استخدام المستخدم
    if not consume_promo_code(code, user_id_str):
        await update.message.reply_text(
            "❌ حدث خطأ أثناء تفعيل الكود.\n"
            "قد يكون:\n"
            "• استخدمته سابقاً\n"
            "• انتهت استخداماته\n"
            "• حدث خطأ تقني"
        )
        return ConversationHandler.END

    # 3. تسجيل المعاملة كـ "bonus" - هذه ستزيد الرصيد تلقائياً
    add_user_history(user_id_str, "bonus", amount, f"promo_{code}")
    
    # 🟢 إضافة سجل في transaction_logs - هذه ستزيد الرصيد
    success = add_transaction(
        user_id=user_id_str,
        tx_type="bonus",
        amount=amount,
        method=f"promo_{code}",
        status="success"
    )
    
    if not success:
        await update.message.reply_text("❌ حدث خطأ أثناء إضافة الرصيد. الرجاء المحاولة لاحقاً.")
        return ConversationHandler.END

    # 🟢 جلب الرصيد الجديد بعد المعاملة
    account_after = get_user_data(user_id_str)
    new_balance = account_after.get('balance', 0.0) if account_after else 0.0

    # 🟢 تحديد نوع الكود للرسالة
    code_type = "خاص" if not is_published else "عام"
    
    # 🟢 عرض عدد المستخدمين الذين استخدموا الكود
    used_by_count = len(get_promo_code_users(code))
    max_uses = promo_data.get('max_uses', 0)
    
    await update.message.reply_text(
        f"✅ مبروك! تم تفعيل الكود ({code_type}) بنجاح.\n\n"
        f"💰 **تم إضافة:** {int(amount):,} ل.س\n"
        f"💳 **رصيدك الجديد:** {int(new_balance):,} ل.س\n\n"
        f"📝 **ملاحظة:** يمكنك استخدام هذا الرصيد لأي عملية في البوت."
    )
    return ConversationHandler.END