# handlers/ishansy/website_withdraw.py
import asyncio
from telegram import Update
from telegram.ext import ContextTypes, ConversationHandler
# 🟢 استيراد دوال SQLite الجديدة
from database import (
    get_user_data, 
    add_transaction, 
    add_user_history,
    # ❌ تم حذف db, save_all_changes, load_db
)
from ishansy_controller import IshansyController
from constants import AWAITING_WEBSITE_WITHDRAW_AMOUNT
from global_state import activate_global_lock


async def _execute_withdraw_in_background(user_id_str: str, username_on_website: str, amount_to_withdraw: float, bot, chat_id_to_reply: int, message_to_edit: int , user_data):
    await activate_global_lock(3) 
    """
    تنفيذ عملية السحب من Ishansy في خيط منفصل وحفظ البيانات وإرسال الرسالة النهائية.
    """
    
    # 1. تنفيذ العملية المتزامنة في خيط منفصل (الاتصال بـ API)
    try:
        success, msg = await asyncio.to_thread(_run_ishansy_withdraw_operation, username_on_website, amount_to_withdraw)
    except Exception as e:
        success = False
        msg = f"❌ خطأ غير متوقع أثناء عملية السحب الخارجي: {e}"

    # 2. تحديث قاعدة البيانات وإعداد الرسالة النهائية
    if success:
        
        # 🟢 التعديل: استخدام add_transaction لإضافة المبلغ إلى الرصيد الأساسي
        transaction_success = add_transaction(
            user_id=user_id_str, 
            tx_type="website_withdraw",  # 🟢 تم إضافتها إلى قائمة العمليات التي تزيد الرصيد
            amount=amount_to_withdraw, # قيمة موجبة للإضافة
            method="site_to_balance", 
            status="success",
            client_number=username_on_website
        )
        
        if not transaction_success:
            # إذا فشلت العملية في قاعدة البيانات
            success = False
            msg = "❌ فشل تحديث الرصيد في قاعدة البيانات"
        else:
            # 🟢 التعديل: تسجيل العملية في سجل المستخدم
            add_user_history(user_id_str, "website_withdraw", amount_to_withdraw, "site_to_balance")

            # 🟢 التعديل: جلب الرصيد الجديد بعد الإضافة
            user_account = get_user_data(user_id_str)
            if user_account:
                new_balance = int(user_account.get('balance', 0.0))
                final_msg = (
                    f"✅ تمت عملية السحب بنجاح!\n\n"
                    f"💰 تم إيداع `{int(amount_to_withdraw):,}` ل.س في محفظتك.\n"
                    f"📊 رصيدك الحالي: `{new_balance:,}` ل.س"
                )
            else:
                final_msg = f"✅ تمت عملية السحب بنجاح! المبلغ: `{int(amount_to_withdraw):,}` ل.س"
    else:
        # ❌ التعامل مع رسالة الفشل
        if "Amount is greater than account balance" in msg or "رصيد غير كافٍ" in msg:
            final_msg = "❌ فشلت عملية السحب: حسابك ليس لديه رصيد كافٍ على الموقع."
        else:
            final_msg = f"❌ فشلت عملية السحب.\n🟥 السبب: {msg}"

    # 3. إرسال الرسالة النهائية للمستخدم (تعديل الرسالة الأولية)
    try:
        # محاولة تعديل رسالة "جاري الانتظار"
        await bot.edit_message_text(
            chat_id=chat_id_to_reply,
            message_id=message_to_edit,
            text=final_msg,
            parse_mode='Markdown'
        )
    except Exception:
        # احتياط: إرسال رسالة جديدة إذا فشل التعديل
        await bot.send_message(
            chat_id=chat_id_to_reply, 
            text=final_msg,
            parse_mode='Markdown'
        )
    finally:
        # ✅ إزالة القفل مهما كانت النتيجة
        user_data.pop('withdraw_lock', None)


async def receive_website_withdraw_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user_id_str = str(update.effective_user.id)

    try:
        amount_to_withdraw = float(update.message.text)
        if amount_to_withdraw <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ مبلغ غير صالح. الرجاء إرسال رقم موجب صحيح.")
        return ConversationHandler.END

    # 🟢 التعديل: استخدام get_user_data
    user_account = get_user_data(user_id_str)
    
    if not user_account:
        await update.message.reply_text("❌ لم يتم العثور على حسابك في البوت.")
        return ConversationHandler.END

    username_on_website = user_account.get('username')
    
    if not username_on_website:
        await update.message.reply_text("❌ لم يتم ربط حسابك باسم مستخدم على الموقع.")
        return ConversationHandler.END

    # 1. إرسال رسالة فورية للمستخدم (للحصول على message_id)
    await_message = await update.message.reply_text(
        f"⏳ جارٍ سحب مبلغ `{int(amount_to_withdraw):,}` ل.س من الموقع، يرجى الانتظار...",
        parse_mode='Markdown'
    )
    
    if context.user_data.get('withdraw_lock'):
        await update.message.reply_text("⚠️ جاري تنفيذ عملية السحب حاليًا. لا يمكنك البدء مرة أخرى.")
        return ConversationHandler.END
    
    context.user_data['withdraw_lock'] = True

    # 2. إطلاق المهمة الخلفية (Fire-and-Forget)
    context.application.create_task(
        _execute_withdraw_in_background(
            user_id_str=user_id_str,
            username_on_website=username_on_website,
            amount_to_withdraw=amount_to_withdraw,
            bot=context.bot,
            chat_id_to_reply=update.effective_chat.id,
            message_to_edit=await_message.message_id, # تمرير ID الرسالة ليتم تعديلها لاحقاً
            user_data=context.user_data
        )
    )

    # 3. إنهاء المحادثة فوراً وتحرير البوت
    return ConversationHandler.END
# ⬅️ دالة متزامنة لتشغيل عملية السحب
def _run_ishansy_withdraw_operation(target_username: str, amount: float) -> tuple[bool, str]:
    """
    تنفيذ عملية السحب باستخدام الـ API الرسمية.
    بدون متصفح – بدون كابتشا.
    """

    controller = IshansyController(username="luffy@gmail.com", password="Aa@723723")

    try:
        # 1️⃣ تسجيل دخول API
        success, resp = controller.login()
        if not success:
            return False, f"❌ فشل تسجيل الدخول: {resp}"

        # 2️⃣ جلب معلومات اللاعب للحصول على playerId
        data = controller.get_player(username=target_username)
        if not data or not data.get("status"):
            return False, "❌ فشل جلب معلومات اللاعب من السيرفر"

        records = data.get("result", {}).get("records", [])
        if not records:
            return False, "❌ اسم اللاعب غير موجود في النظام"

        player_id = records[0]["playerId"]

        # 3️⃣ تنفيذ السحب
        result = controller.withdraw(player_id=player_id, amount=amount)

        if result.get("status") is True:
            return True, f"✅ تم سحب {amount:,.0f} بنجاح من المستخدم {target_username}"

        return False, f"❌ فشل السحب: {result}"

    except Exception as e:
        return False, f"❌ حدث خطأ تقني غير متوقع أثناء السحب: {e}"

async def withdraw_from_website_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    user_id_str = str(query.from_user.id)

    # 🟢 التعديل: استخدام get_user_data
    user_account = get_user_data(user_id_str)
    
    if not user_account or not user_account.get('username'):
        await query.edit_message_text("❌ لم يتم ربط حسابك باسم مستخدم على الموقع. لا يمكن السحب.")
        return ConversationHandler.END

    await query.edit_message_text(
        "💸 سحب الأرباح من الموقع إلى محفظة البوت\n\n"
        "الرجاء إرسال المبلغ الذي تريد سحبه من حسابك على الموقع."
    )
    return AWAITING_WEBSITE_WITHDRAW_AMOUNT


async def receive_website_withdraw_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user_id_str = str(update.effective_user.id)

    try:
        amount_to_withdraw = float(update.message.text)
        if amount_to_withdraw <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ مبلغ غير صالح. الرجاء إرسال رقم موجب صحيح.")
        return ConversationHandler.END
    MIN_WITHDRAW = 15000
    if amount_to_withdraw < MIN_WITHDRAW:
        await update.message.reply_text(f"❌ الحد الأدنى للسحب  هو {MIN_WITHDRAW:,} ل.س.")
        return ConversationHandler.END

    # 🟢 التعديل: استخدام get_user_data
    user_account = get_user_data(user_id_str)
    
    if not user_account:
        await update.message.reply_text("❌ لم يتم العثور على حسابك في البوت.")
        return ConversationHandler.END

    username_on_website = user_account.get('username')

    # 1. إرسال رسالة فورية للمستخدم (للحصول على message_id)
    await_message = await update.message.reply_text(
        f"⏳ جارٍ سحب مبلغ `{int(amount_to_withdraw):,}` ل.س من الموقع، يرجى الانتظار...",
        parse_mode='Markdown'
    )
    if context.user_data.get('withdraw_lock'):
        await update.message.reply_text("⚠️ جاري تنفيذ عملية السحب حاليًا. لا يمكنك البدء مرة أخرى.")
        return ConversationHandler.END
    context.user_data['withdraw_lock'] = True

    # 2. إطلاق المهمة الخلفية (Fire-and-Forget)
    context.application.create_task(
        _execute_withdraw_in_background(
            user_id_str=user_id_str,
            username_on_website=username_on_website,
            amount_to_withdraw=amount_to_withdraw,
            bot=context.bot,
            chat_id_to_reply=update.effective_chat.id,
            message_to_edit=await_message.message_id, # تمرير ID الرسالة ليتم تعديلها لاحقاً
            user_data=context.user_data
        )
    )

    # 3. إنهاء المحادثة فوراً وتحرير البوت
    return ConversationHandler.END


async def cancel_withdraw(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ إزالة القفل في حال تم الإلغاء
    context.user_data.pop('withdraw_lock', None) 
    
    if update.callback_query:
        query = update.callback_query
        await query.answer()
        await query.edit_message_text("❌ تم إلغاء عملية السحب.")
    elif update.message:
        await update.message.reply_text("❌ تم إلغاء عملية السحب.")
    return ConversationHandler.END