import datetime
import asyncio
from datetime import timedelta
from telegram.ext import ContextTypes 
# 🟢 استيراد دوال SQLite
from database import (
    get_last_referral_distribution, # للحصول على تاريخ آخر توزيع
    get_connection,
    set_last_referral_distribution, # لتخزين تاريخ التوزيع الحالي
    process_and_distribute_referral_earnings # الدالة الشاملة التي تنفذ العملية في DB
    # ❌ حذف: db, save_all_changes, load_db, save_data, DB_LOCK
)

DAYS_TO_UNLOCK = 10 

async def process_referral_earnings(context: ContextTypes.DEFAULT_TYPE):
    # ❌ لا حاجة لـ with DB_LOCK: هنا، لأن القفل يتم داخليًا في دالة DB
    
    now = datetime.datetime.now()
    now_iso = now.isoformat()

    # 🌟 التحقق من تاريخ آخر توزيع
    last_dist_str = get_last_referral_distribution()
    
    if last_dist_str:
        last_dist = datetime.datetime.fromisoformat(last_dist_str)
        days_passed = (now - last_dist).days
    else:
        # إذا أول مرة — فقط نخزن التاريخ الحالي ولا نوزع
        set_last_referral_distribution(now_iso)
        return

    # ⛔ لم يحن موعد التوزيع
    if days_passed < DAYS_TO_UNLOCK:
        return

    # --------------------------------------------------
    # 🟢 تنفيذ عملية الدفع والتحديث في DB (كـ معاملة واحدة)
    # --------------------------------------------------
    # هذه الدالة تتولى:
    # 1. جلب الأرباح المعلقة التي تجاوزت DAYS_TO_UNLOCK.
    # 2. التحقق من حالة has_withdrawn للمدعو.
    # 3. إضافة المبلغ إلى 'balance' المُحيل.
    # 4. تحديث حالة سجل الربح إلى 'available'.
    # 5. تسجيل المعاملة (transaction_logs).
    # 6. تصفير حقول referral_pending/available/total للحسابات المتأثرة.
    
    updated_count = process_and_distribute_referral_earnings(now_iso, DAYS_TO_UNLOCK)
    
    # --------------------------------------------------
    # 🕒 تحديث تاريخ آخر توزيع (يتم التحديث فقط إذا تم توزيع أي شيء بنجاح)
    # --------------------------------------------------
    if updated_count > 0:
        set_last_referral_distribution(now_iso)
        print(f"✅ تم توزيع أرباح الإحالة. عدد الأرباح المدفوعة: {updated_count}")

    # ❌ حذف منطق التصفير الفردي وحفظ البيانات لأن كل شيء يتم داخل دالة DB.

async def run_referral_scheduler(context: ContextTypes.DEFAULT_TYPE):
    """
    الدالة التي يتم استدعاؤها بواسطة وظيفة الجدولة (Scheduler Job).
    توزع الأرباح المعلقة كل دقيقة.
    """
    print(f"⏰ [{datetime.datetime.now().strftime('%H:%M:%S')}] بدء تشغيل مجدول الإحالات...")
    
    try:
        # 1. جلب تاريخ آخر توزيع
        last_dist_str = get_last_referral_distribution()
        now = datetime.datetime.now()
        
        if last_dist_str:
            try:
                last_dist = datetime.datetime.fromisoformat(last_dist_str)
                hours_passed = (now - last_dist).total_seconds() / 3600
                print(f"⏳ الساعات المنقضية منذ آخر توزيع: {hours_passed:.2f}")
            except Exception as e:
                print(f"⚠️ خطأ في تحويل تاريخ التوزيع: {e}")
                last_dist = now
        else:
            print("ℹ️ أول مرة يتم تشغيل المجدول")
            last_dist = None
        
        # 2. تنفيذ التوزيع
        now_iso = now.isoformat()
        
        # هنا DAYS_TO_UNLOCK = 10 ولكن للتطوير نجعلها 0 لتوزيع فوري
        DEVELOPMENT_MODE = True  # ❗ اضبط هذا على False للإنتاج
        unlock_days = 0 if DEVELOPMENT_MODE else DAYS_TO_UNLOCK
        
        print(f"🔧 أيام الانتظار المطلوبة: {unlock_days}")
        
        updated_count = process_and_distribute_referral_earnings(now_iso, unlock_days)
        
        # 3. تحديث تاريخ التوزيع إذا تم توزيع أي شيء
        if updated_count > 0:
            set_last_referral_distribution(now_iso)
            print(f"✅ تم توزيع {updated_count} ربح إحالة")
            
            # إرسال إشعار للأدمن
            try:
                from config import ADMIN_IDS
                if ADMIN_IDS:
                    await context.bot.send_message(
                        chat_id=ADMIN_IDS[0],
                        text=f"✅ **توزيع أرباح الإحالة**\n\n"
                             f"📊 تم توزيع: {updated_count} ربح\n"
                             f"⏰ الوقت: {now.strftime('%H:%M:%S')}\n"
                             f"📅 التاريخ: {now.strftime('%Y-%m-%d')}"
                    )
            except Exception as e:
                print(f"⚠️ فشل إرسال إشعار للأدمن: {e}")
        else:
            print(f"ℹ️ لا توجد أرباح مؤهلة للتوزيع في هذا الوقت")
            
        # 4. طباعة إحصائيات
        try:
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("SELECT COUNT(*) as count FROM referral_earnings WHERE status = 'pending'")
                pending_count = cur.fetchone()['count']
                
                cur.execute("SELECT COUNT(*) as count FROM referral_earnings WHERE status = 'available'")
                available_count = cur.fetchone()['count']
                
                print(f"📊 إحصائيات الأرباح: ⏳ معلقة: {pending_count} | ✅ متاحة: {available_count}")
                
        except Exception as e:
            print(f"⚠️ خطأ في جلب الإحصائيات: {e}")
            
    except Exception as e:
        print(f"❌ خطأ في run_referral_scheduler: {e}")


def check_referral_earnings_status():
    """فحص حالة الأرباح للإصلاح"""
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            
            # 1. عدد الأرباح حسب الحالة
            cur.execute("""
                SELECT status, COUNT(*) as count, SUM(amount) as total
                FROM referral_earnings 
                GROUP BY status
            """)
            stats = cur.fetchall()
            
            print("📊 إحصائيات الأرباح:")
            for stat in stats:
                print(f"  {stat['status']}: {stat['count']} ربح = {stat['total'] or 0:.2f}")
            
            # 2. عرض أمثلة
            cur.execute("""
                SELECT id, referrer_id, invited_id, amount, status, created_at
                FROM referral_earnings 
                ORDER BY created_at DESC 
                LIMIT 5
            """)
            examples = cur.fetchall()
            
            print("📋 أمثلة على الأرباح الأخيرة:")
            for ex in examples:
                print(f"  {ex['id']}: {ex['referrer_id']} <- {ex['invited_id']} = {ex['amount']:.2f} ({ex['status']})")
            
            # 3. التحقق من referral_pending في حسابات المستخدمين
            cur.execute("""
                SELECT user_id, referral_pending, referral_available, referral_total
                FROM user_accounts 
                WHERE referral_pending > 0 OR referral_available > 0
                LIMIT 10
            """)
            users = cur.fetchall()
            
            print("👥 مستخدمون لديهم أرباح معلقة/متاحة:")
            for user in users:
                print(f"  {user['user_id']}: ⏳{user['referral_pending']:.2f} | ✅{user['referral_available']:.2f} | 📊{user['referral_total']:.2f}")
                
    except Exception as e:
        print(f"❌ خطأ في فحص الأرباح: {e}")