
import datetime
import asyncio
import sqlite3
from database import (
    get_user_data,
    get_user_net_balance_from_transactions,
    get_referral_relationships,
    get_user_referral_earnings,
    insert_referral_earning,
    get_user_referral_summary,
    update_user_referral_summary,
    has_minimum_active_referrals,
    is_user_active,
    get_last_user_deposit,
    get_connection  # تأكد من وجود هذه الدالة في database.py
)
from config import get_config

async def process_referral_for_deposit(invited_id, deposit_tx ,  bot):
    cfg = get_config()
    ref_cfg = cfg.get("REFERRAL_PERCENTAGES", {})

    REF_PERCENTAGES = [
        float(ref_cfg.get("LEVEL_1", 5)) / 100,
        float(ref_cfg.get("LEVEL_2", 2)) / 100,
        float(ref_cfg.get("LEVEL_3", 1)) / 100
    ]

    relationships = get_referral_relationships()

    # شرط النشاط
    if not is_user_active(invited_id):
        return

    deposit_amount = deposit_tx["amount"]
    deposit_tx_id = deposit_tx["transaction_id"]

    if deposit_amount > 125_000:
        return

    current_invited = invited_id

    for level, percent in enumerate(REF_PERCENTAGES, start=1):
        target_referrer = relationships.get(current_invited)
        if not target_referrer:
            break

        earning_id = f"lvl{level}_{deposit_tx_id}_{target_referrer}"

        # منع التكرار
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute(
                "SELECT 1 FROM referral_earnings WHERE id = ?",
                (earning_id,)
            )
            if cur.fetchone():
                current_invited = target_referrer
                continue

        # شرط الحد الأدنى
        if not has_minimum_active_referrals(target_referrer, minimum=3):
            current_invited = target_referrer
            continue

        earning_amount = round(deposit_amount * percent, 2)
        if earning_amount <= 0:
            current_invited = target_referrer
            continue

        insert_referral_earning({
            "earning_id": earning_id,
            "referrer_id": target_referrer,
            "invited_id": invited_id,
            "amount": earning_amount,
            "status": "pending",
            "level": level,
            "created_at": datetime.datetime.now().isoformat()
        })
        try:
            invited_user = get_user_data(invited_id)
            invited_name = invited_user.get("username") or invited_id

            message = (
                "🎉 <b>أرباح إحالة جديدة</b>\n\n"
                f"💰 تم إضافة <code>{earning_amount:,.0f} ل.س</code>\n"
                f"👤 من إيداع المحال: <code>{invited_name}</code>\n"
                f"📌 المستوى: {level}\n"
                "⏳ الحالة: أرباح معلّقة"
            )

            await bot.send_message(
                chat_id=target_referrer,
                text=message,
                parse_mode="HTML"
            )


        except Exception as e:
            print(f"❌ فشل إرسال إشعار الإحالة: {e}")


        # تحديث الملخص
        all_earnings = get_user_referral_earnings(target_referrer)
        pending = sum(e['amount'] for e in all_earnings if e['status'] == 'pending')
        available = sum(e['amount'] for e in all_earnings if e['status'] == 'available')

        update_user_referral_summary(
            target_referrer,
            pending,
            available,
            pending + available,
            last_processed_deposit_id=deposit_tx_id
        )

        print(f"⚡ إحالة فورية: {earning_amount} للمحيل {target_referrer}")

        current_invited = target_referrer
