# handlers/wallet/transaction_history.py

import time
from telegram import Update
from telegram.ext import ContextTypes
# 🟢 استيراد دوال SQLite الجديدة
from database import get_user_transaction_logs, get_all_transaction_logs 
# ❌ حذف: db, load_db
from config import ADMIN_IDS, PAYMENT_METHODS


# ======================================================
# 👤 عرض سجل المستخدم (عمليات ناجحة فقط)
# ======================================================
async def show_user_transactions(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ حذف db = load_db()
    query = update.callback_query
    await query.answer()
    user_id_str = str(query.from_user.id)

    # 🟢 التعديل: جلب السجلات من SQLite (20 الأخيرة)
    logs = get_user_transaction_logs(user_id_str, status='success', limit=20)
    
    if not logs:
        await query.edit_message_text("❌ لا توجد عمليات ناجحة بعد.")
        return

    text = "📄 سجلك (العمليات الناجحة فقط):\n\n"
    for log in logs:
        t_type = "إيداع" if log['type'] == 'deposit' else "سحب"
        # ⚠️ (الكمية يجب أن تكون من السجل، لكن الحقل في DB هو 'amount')
        amount = int(log.get('amount', 0))
        # ⚠️ (نفترض أن method مخزن كرمز في DB)
        method = PAYMENT_METHODS.get(log.get('method'), log.get('method'))
        # ⚠️ (نفترض أن timestamp مخزن كـ Unix Timestamp)
        timestamp = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(log['timestamp']))
        text += f"{timestamp} | {t_type} | {amount:,} ل.س | {method}\n"

    await query.edit_message_text(text)


# ======================================================
# 👑 عرض كل العمليات للأدمن (ناجحة فقط)
# ======================================================
async def show_all_transactions(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ حذف db = load_db()
    query = update.callback_query
    await query.answer()

    if query.from_user.id not in ADMIN_IDS:
        await query.edit_message_text("❌ هذه الخاصية للأدمن فقط.")
        return

    # 🟢 التعديل: جلب السجلات من SQLite (50 الأخيرة)
    logs = get_all_transaction_logs(status='success', limit=50)

    if not logs:
        await query.edit_message_text("❌ لا توجد عمليات ناجحة بعد.")
        return

    text = "📄 كل العمليات الناجحة:\n\n"
    for log in logs:
        t_type = "إيداع" if log['type'] == 'deposit' else "سحب"
        amount = int(log.get('amount', 0))
        method = PAYMENT_METHODS.get(log.get('method'), log.get('method'))
        user_id = log.get('user_id', 'غير معروف')
        timestamp = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(log['timestamp']))
        text += f"{timestamp} | {t_type} | {amount:,} ل.س | {method} | المستخدم: {user_id}\n"

    await query.edit_message_text(text)


# ======================================================
# 📢 إرسال الإشعار إلى قناة العمليات (الدوال المساعدة)
# ======================================================
# 💡 هذه الدوال لا تصل إلى قاعدة البيانات للقراءة، لكن تم حذف load_db() منها
# حيث أن بيانات المعاملة (transaction) يتم تمريرها كمعامل.

async def send_transaction_to_channel(transaction, context: ContextTypes.DEFAULT_TYPE):
    # ❌ حذف db = load_db()
    from config import OPERATIONS_CHANNEL_ID
    
    t_type = "💵 إيداع" if transaction.get('type') == 'deposit' else "🏧 سحب"
    user_id = transaction.get('user_id', 'غير معروف')
    amount = int(transaction.get('amount', 0))
    method = transaction.get('method', 'غير معروف')
    status = transaction.get('status', 'unknown')
    timestamp = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(transaction.get('timestamp', time.time())))

    status_emoji = "✅" if status in ['success', 'completed'] else "❌"
    text = (
        f"📢 <b>عملية جديدة</b> {status_emoji}\n\n"
        f"🧾 <b>النوع:</b> {t_type}\n"
        f"💰 <b>المبلغ:</b> {amount:,} ل.س\n"
        f"💳 <b>الطريقة:</b> {method}\n"
        f"👤 <b>المستخدم:</b> <code>{user_id}</code>\n"
        f"🕒 <b>التاريخ:</b> {timestamp}"
    )

    await context.bot.send_message(chat_id=OPERATIONS_CHANNEL_ID, text=text, parse_mode="HTML")


def send_transaction_to_channel_via_bot(transaction, bot, job_queue):
    # ❌ حذف db = load_db()
    """إرسال إشعار إلى قناة العمليات باستخدام job_queue من داخل Thread"""
    from config import OPERATIONS_CHANNEL_ID
    # ... (باقي منطق الدالة لا يتغير ويستخدم بيانات المعاملة الممررة) ...
    
    t_type = "💵 إيداع" if transaction.get('type') == 'deposit' else "🏧 سحب"
    user_id = transaction.get('user_id', 'غير معروف')
    amount = int(transaction.get('amount', transaction.get('requested_amount', 0)))
    method = transaction.get('method', 'غير معروف')
    status = transaction.get('status', 'unknown')
    timestamp = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(transaction.get('timestamp', time.time())))

    if status in ['success', 'completed']:
        status_emoji = "✅"
    elif status == 'failed':
        status_emoji = "❌"
    else:
        status_emoji = "⏳"

    text = (
        f"📢 <b>تحديث عملية</b> {status_emoji}\n\n"
        f"🧾 <b>النوع:</b> {t_type}\n"
        f"💰 <b>المبلغ:</b> {amount:,} ل.س\n"
        f"💳 <b>الطريقة:</b> {method}\n"
        f"👤 <b>المستخدم:</b> <code>{user_id}</code>\n"
        f"🕒 <b>التاريخ:</b> {timestamp}"
    )

    # تعريف دالة async داخلية
    async def _send_job(context):
        # ❌ لا حاجة لـ load_db() هنا
        await context.bot.send_message(
            chat_id=OPERATIONS_CHANNEL_ID,
            text=text,
            parse_mode="HTML"
        )

    try:
        job_queue.run_once(_send_job, when=0)
    except Exception as e:
        print(f"❌ فشل جدولة إرسال الإشعار إلى القناة: {e}")


async def send_transaction_to_channel_async(transaction, bot):
    # ❌ حذف db = load_db()
    from config import OPERATIONS_CHANNEL_ID
    # ... (باقي منطق الدالة لا يتغير ويستخدم بيانات المعاملة الممررة) ...
    
    t_type = "💵 إيداع" if transaction.get('type') == 'deposit' else "🏧 سحب"
    user_id = transaction.get('user_id', 'غير معروف')
    amount = int(
        transaction.get('sent_amount') or
        transaction.get('requested_amount') or
        transaction.get('amount', 0)
    )
    method = transaction.get('method', 'غير معروف')
    status = transaction.get('status', 'unknown')
    timestamp = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(transaction.get('timestamp', time.time())))

    status_emoji = "✅" if status in ['success', 'completed'] else "❌" if status == 'failed' else "⏳"

    text = (
        f"📢 <b>تحديث عملية</b> {status_emoji}\n\n"
        f"🧾 <b>النوع:</b> {t_type}\n"
        f"💰 <b>المبلغ:</b> {amount:,} ل.س\n"
        f"💳 <b>الطريقة:</b> {method}\n"
        f"👤 <b>المستخدم:</b> <code>{user_id}</code>\n"
        f"🕒 <b>التاريخ:</b> {timestamp}"
    )

    try:
        await bot.send_message(chat_id=OPERATIONS_CHANNEL_ID, text=text, parse_mode="HTML")
    except Exception as e:
        print(f"❌ فشل إرسال الإشعار إلى القناة: {e}")