import requests
import json

SITE_URL = "https://agents.ichancy.com//"
API_SIGNIN_URL = f"{SITE_URL}global/api/UserApi/signin"
API_REFRESH_URL = f"{SITE_URL}global/api/UserApi/refreshToken"
API_REGISTER_PLAYER = f"{SITE_URL}global/api/UserApi/registerPlayer"
API_GET_PLAYERS = f"{SITE_URL}global/api/UserApi/getPlayersForCurrentAgent"
API_DEPOSIT = f"{SITE_URL}global/api/UserApi/depositToPlayer"
API_WITHDRAW = f"{SITE_URL}global/api/UserApi/withdrawFromPlayer"
API_WALLETS = f"{SITE_URL}global/api/UserApi/getAgentAllWallets"


class IshansyController:

    def __init__(self, username, password):
        self.username = username
        self.password = password
        
        self.access_token = None
        self.refresh_token = None

        self.base_headers = {
            "User-Agent": "Mozilla/5.0",
            "Content-Type": "application/json",
        }

    # ------------------------------------------------------
    #                   تسجيل الدخول الرسمي
    # ------------------------------------------------------
    def login(self):
        payload = {
            "username": self.username,
            "password": self.password
        }

        try:
            response = requests.post(
                API_SIGNIN_URL,
                headers=self.base_headers,
                json=payload
            )

            resp = response.json()

            if resp.get("status") is True:
                self.access_token = resp["result"]["accessToken"]
                self.refresh_token = resp["result"]["refreshToken"]
                return True, resp["result"]

            return False, resp.get("notification", "Login failed")

        except Exception as e:
            return False, f"Error: {e}"

    # ------------------------------------------------------
    #              إضافة التوكن في الهيدر
    # ------------------------------------------------------
    def auth_headers(self):
        return {
            **self.base_headers,
            "Authorization": f"Bearer {self.access_token}"
        }

    # ------------------------------------------------------
    #                   تحديث التوكن
    # ------------------------------------------------------
    def refresh_tokens(self):
        payload = {
            "refreshToken": self.refresh_token
        }

        try:
            response = requests.post(
                API_REFRESH_URL,
                headers=self.base_headers,
                json=payload
            )

            resp = response.json()

            if resp.get("status") is True:
                self.access_token = resp["result"]["accessToken"]
                self.refresh_token = resp["result"]["refreshToken"]
                return True, resp["result"]

            return False, resp

        except Exception as e:
            return False, str(e)

    # ------------------------------------------------------
    #                   إنشاء لاعب
    # ------------------------------------------------------
    def register_player(self, email, password, parent_id, username):
        payload = {
            "player": {
                "email": email,
                "password": password,
                "parentId": parent_id,
                "login": username
            }
        }

        try:
            response = requests.post(
                API_REGISTER_PLAYER,
                headers=self.auth_headers(),
                json=payload
            )
            return response.json()

        except Exception as e:
            return {"error": str(e)}

    # ------------------------------------------------------
    #                 جلب لاعب بالاسم أو ID
    # ------------------------------------------------------
    def get_player(self, player_id=None, username=None):
        if player_id:
            filter_data = {
                "playerId": {
                    "action": "=",
                    "value": player_id,
                    "valueLabel": player_id
                }
            }
        else:
            filter_data = {
                "userName": {
                    "action": "=",
                    "value": username,
                    "valueLabel": username
                }
            }

        payload = {
            "start": 0,
            "limit": 20,
            "filter": filter_data
        }

        try:
            response = requests.post(
                API_GET_PLAYERS,
                headers=self.auth_headers(),
                json=payload
            )
            return response.json()

        except Exception as e:
            return {"error": str(e)}

    # ------------------------------------------------------
    #                       إيداع
    # ------------------------------------------------------
    def deposit(self, player_id, amount, comment=""):
        payload = {
            "amount": float(amount),
            "comment": comment,
            "playerId": player_id,
            "currencyCode": "NSP",
            "currency": "NSP",
            "moneyStatus": 5
        }

        try:
            response = requests.post(
                API_DEPOSIT,
                headers=self.auth_headers(),
                json=payload
            )
            return response.json()

        except Exception as e:
            return {"error": str(e)}

    # ------------------------------------------------------
    #                        سحب
    # ------------------------------------------------------
    def withdraw(self, player_id, amount, comment=""):
        payload = {
            "amount": -abs(float(amount)),  # ALWAYS NEGATIVE
            "comment": comment,
            "playerId": player_id,
            "currencyCode": "NSP",
            "currency": "NSP",
            "moneyStatus": 5
        }

        try:
            response = requests.post(
                API_WITHDRAW,
                headers=self.auth_headers(),
                json=payload
            )
            return response.json()

        except Exception as e:
            return {"error": str(e)}

    # ------------------------------------------------------
    #            جلب رصيد الكاشيرة (المعتمد)
    # ------------------------------------------------------
    def get_cashier_balance(self):
        try:
            response = requests.get(
                API_WALLETS,
                headers=self.auth_headers()
            )
            return response.json()

        except Exception as e:
            return {"error": str(e)}
